
/*
POTAPlus browser extension for POTA.app

Author: David Westbrook K2DW
Contact: dwestbrook@gmail.com or K2DW on POTA Slack workspace.

Support: Direct message or #proj-potaplus-browser-extension channel on POTA Slack workspace

Source & Documentation:
	https://dwestbrook.net/projects/potaplus
*/

var VER = '1.08.01';
// add to idea list:  switch from chrome.storage.local to chrome.storage.sync
// idea list:  add hotkeys .. e.g. L to click QSL button for current spot.
//
/////////////////////////
//
// ==== TODO ====
// TODO: HRD
//   https://code.google.com/archive/p/java-hrd-api/
// 	TCP port 7809
// 	set unblock
// 	set frequency-hz 7200000
// 	set dropdown Mode USB
//		FM AM CW CWR DIG USB LSB
//	set block
//   https://support.hamradiodeluxe.com/support/solutions/articles/51000052684-qso-forwarding
//   	UDP port 2333
//   	N1MM format XML
/////////////////////////
// TODO: ACLOg 7.0.5 (May 2022)
// 	http://www.n3fjp.com/help/api.html
// 	API 2.0 (For programs released after 5/01/2022) -- Added ADDADIFRECORD command.
// 	<CMD><ADDADIFRECORD><VALUE><CALL:4>K2DW<QSO_Date:8>20220317<Time_On:6>205405<Time_Off:6>205405<FREQ:5>7.185<Band:3>40M<Mode:3>SSB<RST_SENT:2>59<RST_RCVD:2>57<SIG_INFO:6>K-1234<MY_SIG_INFO:6>K-5678<Comment:17>This is a Comment<EOR></VALUE></CMD>
//  TODO: Base the QSY selectedcall on spotKey instead of callsign

// TODO: Remember TX PWR by Mode... or make default options
// TODO: ACLog UPDATEANDLOG -- email support to confirm what fields it supports.  COMMENTS? Other1?
// 	May have to instead do the individual text fills, and ENTER command.  see KB3CMT slack chat.
//
// TODO: Send DXCC/STATE  ADIF feilds, based on mapping from region (e.g. US-NY =>  291 & NY)
// TODO: in re-spot dialog, add region label.  Unless pota.app implements it:
// 	https://parksontheair.slack.com/archives/C011E0H540J/p1648904978819709?thread_ts=1648231532.399129&cid=C011E0H540J
// TODO: long term idea "Park heat"   change the color of the park depending on how often its active. this way rarely active parks are one color and the more active parks are different color.
// TODO: flag true ATNO
// 		get grid info at same time
// BUG: reports of bts.historyBtn initialization line having error "anonymous function"
// 	https://parksontheair.slack.com/archives/C02TEHESMLG/p1648984983966939
// BUG: op2op count in wrong place in table view for a VK station.
// 	https://parksontheair.slack.com/archives/C02TEHESMLG/p1649808918243489
// 	https://parksontheair.slack.com/archives/C02TEHESMLG/p1651414919488679
// TODO:  Get the pota.app version ... report in version check? have a max-supported?
// 			Build: 2022-04-17_03:48:37_282
// BUG: in card view,  highlight-new-spots causes QSL/~ btn to disappear
// BUG: card view, darkmode=off,  the QSY doesn't highlight selectedcall.
// TODO: Feature-add in log widow, for a "log as" callsign entry.  e.g.  K2DW/40  multi-radio club op.
// TODO -- test out-of-band frequencies  on CARD mode.
// TODO: q{ on logging (opening of modal??), lookup park info;  Include grid, state, dxcc in log entries }, 
// TODO: N1MM Rotor support
/////////////////////////////////////////////////////


// DEBUG STORAGE:
//chrome.storage.local.get(function(result){console.log(result); console.log("Size MB:", JSON.stringify(result).length/1024/1024)} )


var session_info = getPOTASettings();
var potaSettings = session_info.potaSettings;
var TOKEN = session_info.apiToken;
var THEMECLASS = potaSettings.darkMode ? 'theme--dark' : 'theme--light';
var MYCALL = potaSettings.callsign;
console.log("MYCALL: ", MYCALL);

sendChromeMessage("set__MYCALL", MYCALL);
sendChromeMessage("set__TOKEN", TOKEN);

var HUNTER_CALLS = {};
chrome.runtime.sendMessage({ message: "get__HUNTER_CALLS" }, response => {
      if (response.message === 'success') {
		HUNTER_CALLS = response.payload || {};
		HUNTER_CALLS[ MYCALL ] = Date.now();
		sendChromeMessage("set__HUNTER_CALLS",  HUNTER_CALLS  );
      }
});

var POTA_API_OPTS = {
  headers: {
	  authorization: TOKEN
	, origin: "https://pota.app"
	, accept: "application/json, text/plain, */*"
  }
};

var info = document.createElement('span');
info.innerHTML = `
 <a
	id="upgrade_link"
	target="_blank"
	href="https://dwestbrook.net/projects/potaplus"
	__href="chrome-extension://${chrome.runtime.id}/options.html"
	__href="javascript:runtime.openOptionsPage(); void(0)"
	style="text-decoration:none;
		border-radius: 8px;
		border: 1px solid lightgreen;
	"
	>
 <input id="btn_info" type=button value="POTAPlus Extension" style="
		font-size:smaller !important;
		padding-left: 5px;
		padding-right: 5px;
		padding-top: 2px;
		padding-bottom: 2px;
	"
	_class="ma-2 v-btn v-btn--is-elevated v-btn--has-bg theme--light v-size--default lightgreen"
	class="title text-xs-left primary--text warning"
  />
	<span id="label_ver" style="
		font-family: monospace;
		border-radius: 3px;
		padding: 2px;
		font-size: smaller;
		display: none;
	    ">v${VER}</span>
	<span id="label_update" style="
		background-color: pink;
		font-family: monospace;
		border-radius: 3px;
		padding: 2px;
		font-size: smaller;
		display: none;
	    ">UPDATE</span>
  </a>
`;
var headerObj = Array.from(document.querySelectorAll('span')).find(el => el.textContent.match('Active Spots')).parentElement;
headerObj.classList.remove('col-md-6');
headerObj.classList.add('col-md-10');
headerObj.appendChild(info);
var headerObjCell2 = headerObj.parentElement.querySelector('.col-md-6'); // Find the "At a park now? ADD SPOT" cell in header
if( headerObjCell2 ){
  headerObjCell2.classList.remove('col-md-6');
  headerObjCell2.classList.add('col-md-2');
}

document.querySelector('#label_ver').innerText = "v" + VER;

var MAINTABLE;
var CARDMODE;
var LISTMODE;
var DARKMODE;

var OPTIONS = null;
var LATESTVER = '';
var VERSION_CHECK = '';

function _log(s){
	if( ! OPTIONS.enableLogDisplay ){
	  return;
	}
	console.log("=LOG=", s);
	var box = document.querySelector('#log_container');
	box.style.display = '';
	box.value += s + "\n";
	box.scrollTop = box.scrollHeight;
}
function _fetch(url, opts){
  var which =	  url.match(/api.pota.app/)	? 'POTA'
		: url.match(/dwestbrook.net/)	? 'PLUS'
		: url.match(/http:/)		? 'PROXY'
		:				  'WEB'
  ;
  _log( `[${which}]: ${url}` );
  var label = document.querySelector('#label_update');
  if( ! label.innerText.match(/UPDATE/) ){
	label.style.display = 'none';
  }
  return fetch(url, opts)
	.then(function(response) {
//console.log(response ) ;
		if (!response.ok
//			|| (which=='PROXY' && ! response.payload.success ) // ham-apps-proxy calls are no-cors, so will always be "not ok" with response.status == 0
		    ){
			var s = `FAILED [${response.status}] ${response.statusText} (${which}) ${url}`;
			_log(s);
			_log(response.payload);
			console.log(s);
			console.log(response);
			if( response.status == 403 ){
				document.querySelector('#label_update').innerHTML = 'REFRESH (F5) NEEDED (pota.app session expired)';
				document.querySelector('#label_update').style.display = '';
			}
			// throw Error(response.statusText);
		}
		return response;
	})
	.catch(err => {
			var s = `FAILED: (${which}) ${url} ${err}`;
			_log(s);
			var msg = which == 'PROXY'	? 'Check options, ham-apps-proxy, and rig/logging/rotor apps'
				:  which == 'POTA'	? 'Issue with POTA.app -- try refresh/login'
				:			  'Issue with web API call'
			;
			var label = document.querySelector('#label_update');
			label.style.display = '';
			if( ! label.innerText.match("ERROR") ){
				label.innerHTML = `ERROR -- ${msg}`;
			}
	})
  ;
}

function toggleColor(toggle){
  if( DARKMODE ){
      return  toggle == 1 ? 'blue'
	    : toggle == 2 ? 'grey'
	    : 		    '' ;
  }else{
      return  toggle == 1 ? 'yellow'
	    : toggle == 2 ? 'grey'
	    : 		    '' ;
  }
  return '';
}


function updateVersionLabel() {
	    LATESTVER = VERSION_CHECK.latestVer;
	    if( LATESTVER != VER ){
	      console.log('UPGRADE NEEDED ' + VER + ' => ' + LATESTVER);
	      document.querySelector('#upgrade_link').setAttribute('title', 'Upgrade to v'+LATESTVER);
	      document.querySelector('#label_ver').style.display = '';
	      document.querySelector('#label_update').style.display = '';
	    }
	    MAINTABLE.append(" "); // touch, to trigger initial update
}

function versionCheck() {
  chrome.runtime.sendMessage({ 
        message: "get__VERSION_CHECK"
      }, response => {
      if (response.message === 'success') {
	VERSION_CHECK = response.payload;
//console.log("VERSION_CHECK", VERSION_CHECK);
	if( VERSION_CHECK
		&& VERSION_CHECK.updated
		&& youngerThan(VERSION_CHECK.updated,20)
		&& VERSION_CHECK.installedVer == VER
	    ){
		// use cached info
//console.log("using cached VERSION_CHECK ....");
		updateVersionLabel();
		return;
	}
      }
//console.log("FETCHING NEW VERSION_CHECK INFO....");
      // new version check call
      _log("Version check");
      fetch('https://dwestbrook.net/projects/potaplus/version/?callsign='+MYCALL
	  +'&version='+VER
	  +'&potaSettings='+ encodeURIComponent( JSON.stringify(potaSettings) )
	  +'&options='     + encodeURIComponent( JSON.stringify(OPTIONS) )
      )
      .then(response => response.text())
      .then(data => {
	    data = JSON.parse(data);
	    VERSION_CHECK = {
		    data: data,
		    updated: Date.now(),
		    latestVer: data.version,
		    installedVer: VER,
	    };
	    updateVersionLabel();
	    sendChromeMessage("set__VERSION_CHECK",  VERSION_CHECK  );
      });
  });
}

function clock() {
  document.title = new Date().toISOString().slice(11,19) + " Z";
  setTimeout(clock, 1000);
}

function detectLayout() {
  MAINTABLE = document.querySelector('div.v-data-table__wrapper table tbody');
  CARDMODE = false;
  LISTMODE = true;
  if( MAINTABLE == null ){
	MAINTABLE = document.querySelector('div.layout.row.wrap');
	CARDMODE = true;
	LISTMODE = false;
  }
  DARKMODE = MAINTABLE.closest("div.theme--dark") ? true : false;
}
detectLayout();

chrome.runtime.sendMessage({ 
        message: "get__options"
      }, response => {
      if (response.message === 'success') {
	OPTIONS = setOptionsDefaults( response.payload );
//console.log('OPTIONS', OPTIONS);
	versionCheck();
        if( OPTIONS.enableQsy ){
	    document.getElementById('scan_container').style.display = 'inherit';
        }
	if( OPTIONS.enableMyAwards ){
	    document.querySelector('#myawards_container').style.display = '';
	}
        if( OPTIONS.enableUTCClock ){
		clock();
        }
	if( OPTIONS.enableFilterLabelTimeAbbrv ){
	  Array.from(document.querySelectorAll('b')).find(el => el.textContent.match('Band:'   )).style.display = 'none';
	  Array.from(document.querySelectorAll('b')).find(el => el.textContent.match('Mode:'   )).style.display = 'none';
	  Array.from(document.querySelectorAll('b')).find(el => el.textContent.match('Program:')).style.display = 'none';
	  Array.from(document.querySelectorAll('b')).find(el => el.textContent.match('Sort:'   )).style.display = 'none';
	}
	if( OPTIONS.enableSpaceWX ){
	    document.querySelector('#spacewx_container').style.display = '';
	}
	if( OPTIONS.enableGreyLine ){
	    document.querySelector('#greyline_container').style.display = '';
	}
	if( OPTIONS.enableWidget_RigRef ){
	    document.querySelector('#widget_rigref_container').style.display = '';
	}

	MAINTABLE.append(" "); // touch, to trigger initial update
      }
});

var TOGGLES = '';
chrome.runtime.sendMessage({ message: "get__TOGGLES" }, response => {
      if (response.message === 'success') {
	      TOGGLES = response.payload || {};
      }
});

var TX_PWR = '';
chrome.runtime.sendMessage({ message: "get__TX_PWR" }, response => {
      if (response.message === 'success') {
	      TX_PWR = response.payload || '';
      }
});

var OPS_HUNTED = null;
chrome.runtime.sendMessage({ message: "get__OPS_HUNTED" }, response => {
      if (response.message === 'success') {
	      OPS_HUNTED = response.payload || {};
      }
});

var PARK_INFO = null;
chrome.runtime.sendMessage({ message: "get__PARK_INFO" }, response => {
      if (response.message === 'success') {
	      PARK_INFO = response.payload || {};
      }
});

function updateOpHunted( primaryCall, searchCall, page ){
	var size = 100; // seems like size=100 is the max
	var url = `https://api.pota.app/user/logbook?hunterOnly=1&page=${page}&size=${size}&search=${searchCall}`;
	return _fetch(url, POTA_API_OPTS)
	    .then(response => response.json())
	    .then(data => {
		if( ! (data && data.entries && data.entries.length) ){
			return;
		}
		var ct = 0;
		Array.from( data.entries ).forEach( (s) => {
		  if( s.station_callsign == searchCall ){ // sanity-check against false-positive search matches
			ct++;
		  }
		});
		OPS_HUNTED[primaryCall].ct += ct;
		sendChromeMessage("set__OPS_HUNTED",  OPS_HUNTED  );
		if( data.entries.length && data.count > page*size ){
		  return ct + updateOpHunted( primaryCall, searchCall, page+1 );
		}
		return ct;
	    });
}

var SPOTS_FEED = null;
chrome.runtime.sendMessage({ message: "get__SPOTS_FEED" }, response => {
      if (response.message === 'success') {
	      SPOTS_FEED = response.payload || {};
      }
});

function updateSpotBio( spots ){
	if( ! OPTIONS.enableOpProfileInfo ){
		return;
	}
	if( ! spots ){
		return;
	}
	var div = document.querySelector('#spotbio_container_');
	div.style.fontSize = 'smaller';
	div.style.fontFamily = 'monospace';
	div.style.maxHeight = '200px';
	div.style.overflow = 'scroll';
	var spotsEntries = spots['comments'] || [];
	var parseAlso = {};
	spotsEntries.map( s => s['comments'].match(/{Also:[^}]+}/) ).filter( s => s ).map( s => s[0] ).forEach( s =>  parseAlso[s] = 1 );
	var parseWith = {};
	spotsEntries.map( s => s['comments'].match(/{With:[^}]+}/) ).filter( s => s ).map( s => s[0] ).forEach( s =>  parseWith[s] = 1 );
	div.innerHTML = '';
	if(  Object.keys( parseAlso ).length ){
		div.innerHTML += Object.keys( parseAlso ).sort().join(',') + "<br/>";
		div.innerHTML += '<hr/>';
	}
	if(  Object.keys( parseWith ).length ){
		div.innerHTML += Object.keys( parseWith ).sort().join(',') + "<br/>";
		div.innerHTML += '<hr/>';
	}
	div.innerHTML += spotsEntries
	  .filter( s => s['source']!='RBN' )
	  .map( (s) => {
		// band, comments, frequency, mode, source, spotId, spotTime, spotter
		return [
				  s['spotTime'].match(/T(.....)/)[1].replace(':','')
				, Number(s['frequency'])
				, s['spotter'].padEnd(6,' ').replaceAll(' ','&nbsp;')
				, s['comments']
		].join(' ')
	}).join('<br/>');
}

function updateActivationBio( call, park_num ){
	if( ! OPTIONS.enableOpProfileInfo ){
		return;
	}
	if( ! call || ! park_num ){
		return;
	}
	var div = document.querySelector('#activationbio_container_');
	var data = ACTIVATIONS.find( el =>
		el['activator'] == call
		&& el['reference'] == park_num
	);
	div.innerHTML = !data ? '' : `
		<b>ACTIVATION:</b>
			${data['startDate']}
				${data['startTime']}
			=&gt;
			${ data['startDate'] == data['endDate'] ? '' : data['endDate'] }
				${data['endTime']}
			&nbsp;&nbsp;
			&nbsp;&nbsp;
			<b>QRV:</b> ${data['frequencies']}
			<br/>
		<i>${data['comments']}</i> <br/>
	`;
	/////////
	var spots = SPOTS_FEED[call] ? SPOTS_FEED[call][park_num] : null;
	if( spots ){
		if( youngerThan( spots.updated, 0.1 ) ){ // TODO -- cache length
			updateSpotBio( SPOTS_FEED[call][park_num] );
			return;
		}
	}
/*	{
  "spotId": 3947057,
  "spotTime": "2022-07-10T15:02:52",
  "spotter": "K9LC-#",
  "mode": "FT8",
  "frequency": "14074.0",
  "band": "20m",
  "source": "RBN",
  "comments": "RBN -14 dB via K9LC-#"
} */
    _fetch(`https://api.pota.app/spot/comments/${call}/${park_num}`, POTA_API_OPTS)
    .then(response => response.json())
    .then(data => {
	    SPOTS_FEED[call] ||= {};
	    SPOTS_FEED[call][park_num] = { updated:Date.now(), comments: Array.from(data) };
	    sendChromeMessage("set__SPOTS_FEED",  SPOTS_FEED );
	    updateSpotBio( SPOTS_FEED[call][park_num] );
    }); // fetch
}

var deg2rad = Math.PI / 180;
var rad2deg = 180.0 / Math.PI;
var pi = Math.PI;
var locres = 0; //6 locator characters

function conv_loc_to_deg(locator)
{
 var i = 0;
 var loca = new Array();
 while (i < 10)
  {
  loca[i] = locator.charCodeAt(i) - 65;
  i++;
  }
 loca[2] += 17;
 loca[3] += 17;
 loca[6] += 17;
 loca[7] += 17;
 var lon = (loca[0] * 20 + loca[2] * 2 + loca[4] / 12 + loca[6] / 120 + loca[8] / 2880 - 180);
 var lat = (loca[1] * 10 + loca[3] + loca[5] / 24 + loca[7] / 240 + loca[9] /5760 - 90);
 var geo = { latitude: lat, longitude: lon };
 return(geo);
}

function cos(x)   { return Math.cos(  x); }
function sin(x)   { return Math.sin(  x); }
function acos(x)  { return Math.acos( x); }
function round(x) { return Math.round(x); }

function calc_gc(lat1, lon1, lat2, lon2)
{
var d = acos(sin(lat1) * sin(lat2) + cos(lat1) * cos(lat2) * cos(lon1 - lon2));
var gc_d = round((rad2deg * d) * 60 * 10) / 10;
var gc_dm = round(1.852 * gc_d * 10) / 10;
var gc_ds = round(1.150779 * gc_d * 10) / 10;

if (sin(lon2 - lon1) < 0)
  tc = acos((sin(lat2) - sin(lat1) * cos(d)) / (sin(d) * cos(lat1)));
else if (lon2 - lon1 == 0)
  if (lat2 < lat1)
    tc = deg2rad * 180;
  else
    tc = 0;
else
  tc = 2 * pi - acos((sin(lat2) - sin(lat1) * cos(d)) / (sin(d) * cos(lat1)));

gc_tc = round(tc * rad2deg * 10) / 10;
// Great Circle
// Degrees	Kilometres	Nautical miles	Statute miles
return { deg: gc_tc, km: gc_dm, mi: gc_d, sm: gc_ds };
}


function updateParkBio( p ){
	if( ! OPTIONS.enableOpProfileInfo ){
		return;
	}
	if( ! p || ! p['info'] ){
		return;
	}
	var title = document.querySelector('#parkbio_container_title');
	var div = document.querySelector('#parkbio_container_');
	var divMap = document.querySelector('#parkmap_container_');
	var data = p['info'];
	var stats = p['stats'];
	var mapZoom = 3;
	var mapDim = "100x100";  // W x H pixels
	var longLat = data['longitude'] + ',' + data['latitude'];
	var latLong = data['latitude']  + ',' + data['longitude'];
	// TODO badges ... park hunted ct;  reference count
	title.innerHTML = `
		<span style="font-size:larger">
			${data['reference']}
				${data['name']}
				${data['parktypeDesc']}
				<br/>
		</span>
	`;
	var myGrid = OPTIONS.loggingConfig.homeGrid || '';
	var parkGrid = data['grid6'] || '';
	var distanceInfo = '';  // for distance and bearing HTML
	if( myGrid.length == 6 && parkGrid.length == 6 ){
		var loc1 = myGrid   + '55AA';
		var loc2 = parkGrid + '55AA';

		var geo1 = conv_loc_to_deg(loc1);
		var geo2 = conv_loc_to_deg(loc2);

		var lon1 = geo1.longitude * deg2rad;
		var lat1 = geo1.latitude * deg2rad;

		var lon2 = geo2.longitude * deg2rad;
		var lat2 = geo2.latitude * deg2rad;

		var greatCircle = calc_gc(lat1, -lon1, lat2, -lon2);
		distanceInfo = `
			<div style="
				border: 2px solid black;
				width: fit-content;
				border-radius: 5px;
				padding: 2px;
				margin-left: 25px;
				display: inline;
				">
				&nbsp;&nbsp;
				${round( greatCircle.deg )}&deg;
				&nbsp;&nbsp;
				${round( greatCircle.mi  )} miles
				&nbsp;&nbsp;
			</div>
		`;
		if( OPTIONS.enableRotor ){
			var url;
			var baseURL = OPTIONS.rotorConfig.proto + "://" + OPTIONS.rotorConfig.host + ":" + OPTIONS.rotorConfig.port;
			var baseParams = '';
			if( OPTIONS.rotorConfig.target_host ){ baseParams += '&__host='+OPTIONS.rotorConfig.target_host; }
			if( OPTIONS.rotorConfig.target_port ){ baseParams += '&__port='+OPTIONS.rotorConfig.target_port; }
			if( OPTIONS.rotorConfig.method == 'pst' ){
		          url = baseURL + "/pst?QRA=" + parkGrid + baseParams;
			}
			distanceInfo += `
				&nbsp;
				<span id="rotate_container"
				  class="ma-2 v-btn v-btn--is-elevated v-btn--has-bg theme--light v-size--default secondary"
				  style="
					border-radius: 5px;
					padding: 6px;
					border: 1px solid grey;
					white-space: nowrap;
					"
				  >
					<input id="btn_rotate" type=button value="ROTATE" data-url="${url}" />
				</span>
			`;
		}
	}

	divMap.innerHTML = `
		<iframe width="275" height="175"
			src="https://www.google.com/maps/embed/v1/streetview?key=AIzaSyABwjeMlP3FIw9kHNAA2Wn8WHyQR2cUkjk&location=${latLong}"
			></iframe>
	`;
	div.innerHTML = `
		<a target="_blank"
			style="margin-right:20px; float:left"
			href="https://pota.app/#/park/${data['reference']}"
		><img id="park_map_img"
			src="https://api.pota.app/styles/v1/mapbox/outdoors-v11/static/pin-l-park+f74e4e(${longLat})/${longLat},${mapZoom},0,0/${mapDim}?access_token=pk.eyJ1IjoibjBhdyIsImEiOiJjand6aW5vMHoxZ29qNDluNW1rcnQ1Ymw3In0.Maco-UBPruDo_Gh13oBKPw"
		></a>
		<span style="font-size:smaller">
			${data['grid6']}
				${data['locationDesc']}
				${data['entityName'].replace('United States Of America','U.S.A')}
				<br/>
			${stats['activations']} Activations
				<!-- ${stats['attempts']} -->
				${stats['contacts']} qsos<br/>
			${distanceInfo}
		</span>
	`;
	if( OPTIONS.enableRotor ){
		document.querySelector('#btn_rotate').addEventListener('click', (e2) => {
			var url = e2.target.getAttribute('data-url');
			_fetch( url, { mode: 'no-cors' } );
		} );
	}
}

function updateBio( u ){
	if( ! OPTIONS.enableOpProfileInfo ){
		return;
	}
	if( ! u || ! u['bio'] || ! u['bio']['callsign'] ){
		return;
	}
	var data = u['bio'];
	var otherCalls = data['other_callsigns'] || [];
	var hunter = data['stats']['hunter'] || {};
	var attempts = data['stats']['attempts'] || {};
	var div = document.querySelector('#bio_container_');
	var title = document.querySelector('#bio_container_title');
	var op2opBadge = OPTIONS.enableNeededOps ?
		`<span data-marker="OPS_HUNTED" class="v-badge__badge ${ u['ct'] ? 'accent' : 'warning' }" title="Total QSOs with ${data['callsign']}" style="display: NONE; position: relative; top:-3px">${u['ct']}</span>&nbsp;`
		: '';
	title.innerHTML = `
		<span style="font-size:larger"> ${data['callsign']} </span>
			${op2opBadge}
			<span style="font-size:larger"> ${data['name']} </span>
			<br/>
	`;
	div.innerHTML = `
		<div style="height:110px">
		    <a target="_blank"
			style="vertical-align: middle; margin-right:20px; float:left"
			href="https://pota.app/#/profile/${data['callsign']}"
		    ><img id="bio_container_img"
			src="https://www.gravatar.com/avatar/${data['gravatar']}?s=100&amp;d=identicon"
		    ></a>
				${otherCalls.join('<br/>')}
		</div>
		<span style="font-size:smaller">
			<!-- ${data['stats']['awards']} awards ${data['stats']['endorsements']} endorsements	<br/> -->
			Hunted ${hunter['parks']} parks ${hunter['qsos']} qsos	<br/>
			${attempts['activations']} Activations ${attempts['parks']} parks ${attempts['qsos']} qsos	<br/>
			${data['qth'] ? "Home QTH: "+data['qth'] : ''} <br/>
		</span>
	`;
/*
	/////////////////////////////////////////////
	var box = document.querySelector('#bio_container');
	box.value = '';
	box.style.display = '';
	var img = document.querySelector('#bio_container_img');
	img.src = '';
	img.style.display = '';
	var data = u['bio'];
	data['stats'] ||= {};
	var otherCalls = data['other_callsigns'] || [];
	img.src = `https://www.gravatar.com/avatar/${data['gravatar']}?s=100&d=identicon`;
	box.value += data['callsign'] + "       " + otherCalls.join(', ') + "\n";
	box.value += data['name'] + "\n";
//	box.value += data['qth'] + "\n";
//	box.value += data['stats']['awards'] + ' awards ' + data['stats']['endorsements'] + ' endorsements' + "\n";
	var hunter = data['stats']['hunter'] || {};
	box.value += 'Hunted ' + hunter['parks'] + ' parks ' + hunter['qsos'] + ' qsos' + "\n";
	var attempts = data['stats']['attempts'] || {};
	box.value += attempts['activations'] + ' Activations ' + attempts['parks'] + ' parks ' + attempts['qsos'] + ' qsos' + "\n";
*/
}

function lookupOpHunted (call) {
	if( ! OPTIONS.enableNeededOps ){
		updateBio(OPS_HUNTED[call]);
		return;
	}
	if( ! call ){
		// not sure how this happens, but it seems to ... so bail, just in case...
		return;
	}
	if( OPS_HUNTED[call] && OPS_HUNTED[call]['bio'] ){
		if( youngerThan( OPS_HUNTED[call].updated, 20 ) ){
			updateBio(OPS_HUNTED[call]);
			return;
		}
	}
	OPS_HUNTED[call] ||= { "ct": 0, "updated": Date.now(), "bio":{} };
	OPS_HUNTED[call]['updated'] = Date.now();
	var searchCall = call.replace( /\/.*/, '' );
	return _fetch('https://api.pota.app/profile/'+searchCall, POTA_API_OPTS)
	    .then(response => response.json())
	    .then(data => {
		if( ! (data && data['callsign'] )){
			return;
		}
		var primaryCall = data['callsign'];
		var otherCalls = data['other_callsigns'];
// awards	callsign	gravatar	id	name	other_callsigns	qth	recent_activity{}	stats.activator{activations,parks,qsos}	stats.attempts{activations,parks,qsos}	stats.awards	stats.endorsements	stats.hunter{parks,qsos}
		    // https://www.gravatar.com/avatar/98c8065d8145a944c1825697f6a600c8?s=250&d=identicon
//console.log('LOOKUP OPS HUNTED data', searchCall, primaryCall, otherCalls);
		OPS_HUNTED[primaryCall] = { "ct": 0, "updated": Date.now(), "bio": data };
		return Promise.all(
		  [ Promise.resolve( updateOpHunted(primaryCall, primaryCall, 1) ) ].concat(
			otherCalls.map( c => updateOpHunted(primaryCall, c, 1) )
		  )
		).then( (values) => {
//console.log('OP2OP', primaryCall, OPS_HUNTED[primaryCall], otherCalls, values );
		  for (let c of otherCalls) {
//console.log('OP2OP-alts', primaryCall, c, OPS_HUNTED[primaryCall] );
		    OPS_HUNTED[c] = OPS_HUNTED[primaryCall];
		  }
		  updateBio(OPS_HUNTED[primaryCall]);
		  MAINTABLE.append(" "); // touch, to trigger initial update
		});
	    });
}

function lookupParkHunted (park_num) {
	if( ! OPTIONS.enableNeededOps ){
		updateParkBio(PARK_INFO[park_num]);
		return;
	}
	if( ! park_num ){
		// not sure how this happens, but it seems to ... so bail, just in case...
		return;
	}
	if( PARK_INFO[park_num] && PARK_INFO[park_num]['info'] ){
		if( youngerThan( PARK_INFO[park_num].updated, 20 ) ){
			updateParkBio(PARK_INFO[park_num]);
			return;
		}
	}
	PARK_INFO[park_num] ||= { "ct": 0, "updated": Date.now(), "info":{}, "stats":{} };
	PARK_INFO[park_num]['updated'] = Date.now();
	return _fetch('https://api.pota.app/park/'+park_num, POTA_API_OPTS)
	// https://api.pota.app/park/K-6532
	// https://api.pota.app/park/stats/K-6532
	    .then(response => response.json())
	    .then(data => {
		if( ! (data && data['reference'] )){
			return;
		}
		var park_ref = data['reference'];
		//
		// https://parksontheair.com/pota-awards/#advanced-awards
		// https://docs.pota.app/assets/images/shift_map.png
		var lng = data['longitude'];
		data['lateshift_start' ] = ( Math.round(18 - lng / 15.0)      % 24).toString().padStart(2,'0').padEnd(4,'0');
		data['lateshift_end'   ] = ((Math.round(18 - lng / 15.0) + 8) % 24).toString().padStart(2,'0').padEnd(4,'0');
		data['earlyshift_start'] = ( Math.round(26 - lng / 15.0)      % 24).toString().padStart(2,'0').padEnd(4,'0');
		data['earlyshift_end'  ] = ((Math.round(26 - lng / 15.0) + 6) % 24).toString().padStart(2,'0').padEnd(4,'0');
		//
		PARK_INFO[park_ref] = { "ct": 0, "updated": Date.now(), "info": data, "stats":{} };
		return _fetch('https://api.pota.app/park/stats/'+park_num, POTA_API_OPTS)
		    .then(response => response.json())
		    .then(data => {
			if( ! (data && data['reference'] )){
				return;
			}
			var park_ref = data['reference'];
			PARK_INFO[park_ref]['stats'] = data;
			    // write PARK_INFO to cache.
			sendChromeMessage("set__PARK_INFO",  PARK_INFO );
			updateParkBio( PARK_INFO[park_ref] );
		    });
	    });
}


function _clean_N1CC_HUNTED_item(hunted){
	hunted["bandsWorked"]    ||= {};
	hunted["bandsWorkedStr"] ||= '';

	if( hunted["bandsWorked"].hasOwnProperty('INVALID') ){
		delete hunted["bandsWorked"]['INVALID'];
	}
	hunted["bandsWorkedStr"] = Object.keys( hunted["bandsWorked"] )
		.sort( (a,b) => compareBands(a,b) )
		.join(',');
	hunted["bandsCtCached"]   = Object.keys( hunted["bandsWorked"] ).length;
	Object.keys( hunted["bandsWorked"] ).forEach( band => {
		// convert any old storage from QSO object to simple boolean.
		if( typeof( hunted["bandsWorked"][band] ) == 'object' ){
			hunted["bandsWorked"][band] = hunted["bandsWorked"][band].qsoId;
		}
	} );
	hunted["bandsCtOfficial"] = hunted["bands"];
	hunted["cacheComplete"] = hunted["bandsCtOfficial"] && ( hunted["bandsCtCached"] == hunted["bandsCtOfficial"] );
	return hunted;
}

function updateMyAwardsStat ( k, currentNum, nextLevel ){
	var numerator   = document.querySelector('#myawards_container').querySelector(`#${k}_hunted`);
	var denominator = document.querySelector('#myawards_container').querySelector(`#${k}_next`);
	numerator.innerHTML = currentNum;
	denominator.innerHTML = nextLevel;
	if( nextLevel - currentNum <= 10 ){
	  Object.assign( numerator.style, {
		"background-color": "orange",
		"padding": "5px",
		"font-weight": "bold",
		"border-radius": "8px",
		"font-size": "larger",
	  });
	}
	return true;
}

var WAS_HUNTED = null;
var PARKS_HUNTED = null;
var ACTIVATIONS = null;
var N1CC_HUNTED = null;
var LOCATIONS_HUNTED = null;
//
var MYAWARDS_PARKS = null;
var MYAWARDS_LATESHIFT = null;
var MYAWARDS_EARNED = {};
var SPACEWX = {};
if( MYCALL){
  ////////
  chrome.runtime.sendMessage({ message: "get__LOCATIONS_HUNTED_cache" }, response => {
    if (response.message === 'success') {
      var cache = response.payload;
      if( cache && cache.data && cache.updated && youngerThan(cache.updated,20)){
	LOCATIONS_HUNTED = cache.data;	// use cached
	return;
      }
    }
    if( ! OPTIONS.enableNeededLocations ){
	LOCATIONS_HUNTED = [];  // set to non-null so main handler thinks it is loaded. 
	return;
    }
    // refresh from pota.app API
    _fetch('https://api.pota.app/user/award/progress/hunter/allrefs', POTA_API_OPTS)
    .then(response => response.json())
    .then(data => {
	    LOCATIONS_HUNTED = Array.from( data );
	    sendChromeMessage("set__LOCATIONS_HUNTED_cache",  {updated:Date.now(), data:LOCATIONS_HUNTED}  );
	    MAINTABLE.append(" "); // touch, to trigger initial update
	    //console.log("Locations CT", LOCATIONS_HUNTED.length);
	    //console.log("Locations MEM SIZE", JSON.stringify(LOCATIONS_HUNTED).length);
	    //console.log(LOCATIONS_HUNTED[0]);
/*
  {
 			locationDesc: "US-DC"
			locationId: 9
			locationName: "District Of Columbia"
			numParksActivated: 9
			numParksAvailable: 30
			toGo: 21
  },
*/
    }); // fetch
  });  // get__LOCATIONS_HUNTED_cache
  ////////
  chrome.runtime.sendMessage({ message: "get__WAS_HUNTED_cache" }, response => {
    if (response.message === 'success') {
      var cache = response.payload;
      if( cache && cache.data && cache.updated && youngerThan(cache.updated,20)){
	WAS_HUNTED = cache.data;	// use cached
	return;
      }
    }
    if( ! OPTIONS.enableNeededWAS ){
	WAS_HUNTED = [];  // set to non-null so main handler thinks it is loaded. 
	return;
    }
    // refresh from pota.app API
    _fetch('https://api.pota.app/user/award/progress/hunter/usstates', POTA_API_OPTS)
    .then(response => response.json())
    .then(data => {
	    WAS_HUNTED = Array.from( data );
	    sendChromeMessage("set__WAS_HUNTED_cache",  {updated:Date.now(), data:WAS_HUNTED}  );
	    MAINTABLE.append(" "); // touch, to trigger initial update
	    //console.log("WAS CT", WAS_HUNTED.length);
	    //console.log("WAS MEM SIZE", JSON.stringify(WAS_HUNTED).length);
	    //console.log(WAS_HUNTED[0]);
/*
  {
    "locationId": 3,
    "locationDesc": "US-AL",
    "locationName": "Alabama"
  },
*/
    }); // fetch
  });  // get__WAS_HUNTED_cache
  ////////
  chrome.runtime.sendMessage({ message: "get__PARKS_HUNTED_cache" }, response => {
    if (response.message === 'success') {
      var cache = response.payload;
      if( cache && cache.data && cache.updated && youngerThan(cache.updated,20)){
	PARKS_HUNTED = cache.data;	// use cached
	return;
      }
    }
    if( ! OPTIONS.enableNeededParks && ! OPTIONS.enableNeededLocations ){
	PARKS_HUNTED = [];  // set to non-null so main handler thinks it is loaded. 
	return;
    }
    // refresh from pota.app API
    _fetch('https://api.pota.app/user/stats/hunter/park', POTA_API_OPTS)
    .then(response => response.json())
    .then(data => {
	    PARKS_HUNTED = Array.from( data );
	    sendChromeMessage("set__PARKS_HUNTED_cache",  {updated:Date.now(), data:PARKS_HUNTED}  );
	    MAINTABLE.append(" "); // touch, to trigger initial update
	    //console.log("PARKS CT", PARKS_HUNTED.length);
	    //console.log("PARKS MEM SIZE", JSON.stringify(PARKS_HUNTED).length);
	    //console.log(PARKS_HUNTED[0]);
/*
  {
    "entity": "Canada",
    "location": "New Brunswick",
    "short": "CA-NB",
    "reference": "VE-0786",
    "park": "Meredith Houseworth Memorial Seashore Natural Area",
    "first": "2020-07-18T00:31:10",
    "qsos": 2
  },
*/
    }); // fetch
  });  // get__PARKS_HUNTED_cache
  ////////
  chrome.runtime.sendMessage({ message: "get__N1CC_HUNTED_cache" }, response => {
    if (response.message === 'success') {
      var cache = response.payload;
      N1CC_HUNTED = ( cache ? cache.data : null ) || [];
      var n = JSON.stringify( N1CC_HUNTED ).length;
      N1CC_HUNTED = N1CC_HUNTED.map( (hunted) => _clean_N1CC_HUNTED_item(hunted) );
      if( cache && cache.data && cache.updated && youngerThan(cache.updated,12) ){
	return; // use cached
      }
    }
    if( ! OPTIONS.enableNeededN1CC ){
	N1CC_HUNTED = [];  // set to non-null so main handler thinks it is loaded. 
	return;
    }
    _fetch('https://api.pota.app/user/award/progress/hunter/n1ccmultiband', POTA_API_OPTS) // refresh from pota.app API
    .then(response => response.json())
    .then(data => {
	var fetched = Array.from( data );
	fetched.forEach( (summaryItem) => {
		// effectively upsert into N!CC_HUNTED array
		var hunted = N1CC_HUNTED.find(x => x.reference == summaryItem.reference);
		if(hunted){
			hunted.bands = summaryItem.bands;
			hunted.name = summaryItem.name;
		}else{
			N1CC_HUNTED.push(summaryItem);
		}
	});
	N1CC_HUNTED = N1CC_HUNTED.map( (hunted) => _clean_N1CC_HUNTED_item(hunted) );
	// initial save, to cache the hunter/n1ccmultiband endpoint check
	sendChromeMessage("set__N1CC_HUNTED_cache",  {updated:Date.now(), data:N1CC_HUNTED}  );
	MAINTABLE.append(" "); // touch, to trigger initial update
	N1CC_HUNTED.forEach( (hunted) => {
	    var park_num = hunted["reference"];
	    if( ! hunted["cacheComplete"] ){
	      _updateBandsWorked(1, park_num, hunted).then( (x)=>{
		sendChromeMessage("set__N1CC_HUNTED_cache",  {updated:Date.now(), data:N1CC_HUNTED}  );
		MAINTABLE.append(" "); // touch, to trigger initial update
	      });
	    }
	});
	    //console.log("N1CC CT", N1CC_HUNTED.length);
	    //console.log("N1CC MEM SIZE", JSON.stringify(N1CC_HUNTED).length);
	    //console.log(N1CC_HUNTED[0]);
/*
  {
    "reference": "K-8071",
    "name": "Mohawk River State Park",
    "bands": 3
    , ... // custom attributes from _clean_N1CC_HUNTED_item(hunted);
  },
*/
//N1CC_HUNTED = [ { "reference": "K-2726", "name": "MOCK DATA", "bands": 8 }, { "reference": "K-6706", "name": "MOCK DATA", "bands": 4 }, ]; // DEBUG mock data
/*
N1CC_HUNTED = [
	{ "reference": "K-1909", "name": "MOCK DATA", "bands": 8 }
	, { "reference": "K-2261", "name": "MOCK DATA", "bands": 6 }
	, { "reference": "K-3430", "name": "MOCK DATA", "bands": 3 }
//	, { "reference": "K-2261", "name": "MOCK DATA", "bands": 3 }
	, ]; // DEBUG mock data
*/
    }); // fetch
  });  // get__N1CC_HUNTED_cache
  ////////
  chrome.runtime.sendMessage({ message: "get__MYAWARDS_PARKS_cache" }, response => {
    if( ! OPTIONS.enableMyAwards ){
	return;
    }
    if (response.message === 'success') {
      var cache = response.payload;
      if( cache && cache.data && cache.updated && youngerThan(cache.updated,20)){
	MYAWARDS_PARKS = cache.data;	// use cached
	updateMyAwardsStat( "parks", MYAWARDS_PARKS['numParks'], MYAWARDS_PARKS['nextAwardLevel'] );
//	document.querySelector('#myawards_container').querySelector('#parks_hunted').innerHTML = MYAWARDS_PARKS['numParks'];
//	document.querySelector('#myawards_container').querySelector('#parks_next').innerHTML = MYAWARDS_PARKS['nextAwardLevel'];
//	if( MYAWARDS_PARKS['nextAwardLevel'] - MYAWARDS_PARKS['numParks'] <= 10 ){
//	  Object.assign( document.querySelector('#myawards_container').querySelector('#parks_hunted').style, {
//		"background-color": "orange",
//		"padding": "5px",
//		"font-weight": "bold",
//		"border-radius": "8px",
//		"font-size": "larger",
//	  });
//	}
	return;
      }
    }
    _fetch('https://api.pota.app/user/award/progress/hunter/parks', POTA_API_OPTS) // refresh from pota.app API
    .then(response => response.json())
    .then(data => {
	    MYAWARDS_PARKS = data;
	    //  {"numParks": 1169, "nextAwardLevel": 1500, "completedPct": 77.93333333333334}
	    sendChromeMessage("set__MYAWARDS_PARKS_cache",  {updated:Date.now(), data:MYAWARDS_PARKS}  );
	    document.querySelector('#myawards_container').querySelector('#parks_hunted').innerHTML = MYAWARDS_PARKS['numParks'];
	    document.querySelector('#myawards_container').querySelector('#parks_next').innerHTML = MYAWARDS_PARKS['nextAwardLevel'];
    }); // fetch
  });  // get__MYAWARDS_PARKS_cache
  ////////
  chrome.runtime.sendMessage({ message: "get__MYAWARDS_LATESHIFT_cache" }, response => {
    if( ! OPTIONS.enableMyAwards ){
	return;
    }
    if (response.message === 'success') {
      var cache = response.payload;
      if( cache && cache.data && cache.updated && youngerThan(cache.updated,20)){
	MYAWARDS_LATESHIFT = cache.data;	// use cached
	updateMyAwardsStat( "lateshift", MYAWARDS_LATESHIFT['contacts'], MYAWARDS_LATESHIFT['nextAwardLevel'] );
//	document.querySelector('#myawards_container').querySelector('#lateshift_hunted').innerHTML = MYAWARDS_LATESHIFT['contacts'];
//	document.querySelector('#myawards_container').querySelector('#lateshift_next').innerHTML = MYAWARDS_LATESHIFT['nextAwardLevel'];
	return;
      }
    }
    _fetch('https://api.pota.app/user/award/progress/hunter/lateshift', POTA_API_OPTS) // refresh from pota.app API
    .then(response => response.json())
    .then(data => {
	    MYAWARDS_LATESHIFT = data;
	    //  {"contacts": 636, "nextAwardLevel": 650, "completedPct": 97.84615384615384}
	    sendChromeMessage("set__MYAWARDS_LATESHIFT_cache",  {updated:Date.now(), data:MYAWARDS_LATESHIFT}  );
	    document.querySelector('#myawards_container').querySelector('#lateshift_hunted').innerHTML = MYAWARDS_LATESHIFT['contacts'];
	    document.querySelector('#myawards_container').querySelector('#lateshift_next').innerHTML = MYAWARDS_LATESHIFT['nextAwardLevel'];
    }); // fetch
  });  // get__MYAWARDS_LATESHIFT_cache
  ////////
  chrome.runtime.sendMessage({ message: "get__MYAWARDS_EARNED_cache" }, response => {
    if( ! OPTIONS.enableMyAwards ){
	return;
    }
    if (response.message === 'success') {
      var cache = response.payload;
      MYAWARDS_EARNED = cache ? JSON.parse(cache.data) || {} : {};
//DEBUG>>>//
//MYAWARDS_EARNED['prev'] = MYAWARDS_EARNED['curr'].slice(1,35);
//MYAWARDS_EARNED['delta'] = MYAWARDS_EARNED['curr'].filter(x => MYAWARDS_EARNED['prev'].indexOf(x) === -1);
//DEBUG<<<//
//console.log('MYAWARDS_EARNED', MYAWARDS_EARNED);
      if( cache && cache.data && cache.updated && youngerThan(cache.updated,20)){
	    if( MYAWARDS_EARNED['delta'] && MYAWARDS_EARNED['delta'].length && MYAWARDS_EARNED['prev'].length ){ 
		var earnedHTML = `New awards earned!!<br/> <ul>`;
		earnedHTML += MYAWARDS_EARNED['delta'].map( (s) => {
		  return '<li>' + [
				 s['dateGranted']
				,s['awardName']
				,s['badgeNum']
				,s['endorsements']
				,s['reference']
				,s['operator']
				,s['radarDate']
				,s['additionalDetails']
				,s['awardDescription']
		  ].filter(i=>i).join(' | ') + '</li>';
		}).sort().reverse().join('\n');
		earnedHTML += `</ul>`;
		document.querySelector('#earned_container').innerHTML = earnedHTML;
		document.querySelector('#earned_container').style.display = '';
	    }
	    return;
      }
    }
    _fetch('https://api.pota.app/user/award/earned', POTA_API_OPTS) // refresh from pota.app API
    .then(response => response.json())
    .then(data => {
	    MYAWARDS_EARNED['prev'] = (MYAWARDS_EARNED['curr'] || []).map( (x) => JSON.stringify(x) );
	    MYAWARDS_EARNED['curr'] = Array.from( data ) || [];
// DEBUGGING // MYAWARDS_EARNED['prev'].pop(); MYAWARDS_EARNED['prev'].pop();
	    MYAWARDS_EARNED['delta'] = MYAWARDS_EARNED['curr'].filter(x => MYAWARDS_EARNED['prev'].indexOf( JSON.stringify(x) ) === -1);
//console.log( MYAWARDS_EARNED );
//  [ {"activeCallsign": "K2DW", "dateGranted": "2022-05-22T06:55:00", "awardName": "Silver Activator", "awardDescription": "Working from 20 different units", "badgeNum": null, "endorsements": "PHONE", "reference": null, "parkName": null, "operator": null, "radarDate": null, "additionalDetails": null}, ... ]
	    sendChromeMessage( "set__MYAWARDS_EARNED_cache",  {updated:Date.now(), data:JSON.stringify(MYAWARDS_EARNED)} );
	    if( MYAWARDS_EARNED['delta'] && MYAWARDS_EARNED['delta'].length && MYAWARDS_EARNED['prev'].length ){ 
		var earnedHTML = `New awards earned!!<br/> <ul>`;
		earnedHTML += MYAWARDS_EARNED['delta'].map( (s) => {
		  return '<li>' + [
				 s['dateGranted']
				,s['awardName']
				,s['badgeNum']
				,s['endorsements']
				,s['reference']
				,s['operator']
				,s['radarDate']
				,s['additionalDetails']
		  ].filter(i=>i).join(' | ') + '</li>';
		}).sort().reverse().join('\n');
		earnedHTML += `</ul>`;
		document.querySelector('#earned_container').innerHTML = earnedHTML;
		document.querySelector('#earned_container').style.display = '';
	    }
    }); // fetch
  });  // get__MYAWARDS_EARNED_cache
  ////////
  chrome.runtime.sendMessage({ message: "get__SPACEWX_FLUX_cache" }, response => {
    if( ! OPTIONS.enableSpaceWX ){
	return;
    }
    if (response.message === 'success') {
      var cache = response.payload;
      if( cache && cache.data && cache.updated && youngerThan(cache.updated,12)){
	SPACEWX['flux'] = cache.data;	// use cached
	document.querySelector('#spacewx_container').querySelector('#sfi').innerHTML = SPACEWX['flux']['flux'];
	document.querySelector('#spacewx_container').querySelector('#sfi').setAttribute('title', SPACEWX['flux']['time_tag']);
	return;
      }
    }
    _fetch('https://services.swpc.noaa.gov/json/f107_cm_flux.json') 
    .then(response => response.json())
    .then(data => {
	    SPACEWX['flux'] = data[0];
	    // {"time_tag":"2022-06-05T22:00:00","frequency":2800,"flux":9.500000000000000e+001,"reporting_schedule":"Afternoon","avg_begin_date":null,"ninety_day_mean":null,"rec_count":null}
	    sendChromeMessage("set__SPACEWX_FLUX_cache",  {updated:Date.now(), data: SPACEWX['flux']}  );
	    document.querySelector('#spacewx_container').querySelector('#sfi').innerHTML = SPACEWX['flux']['flux'];
	    document.querySelector('#spacewx_container').querySelector('#sfi').setAttribute('title', SPACEWX['flux']['time_tag']);
    }); // fetch
  });  // get__SPACEWX_FLUX_cache
  ////////
  chrome.runtime.sendMessage({ message: "get__SPACEWX_KA_cache" }, response => {
    if( ! OPTIONS.enableSpaceWX ){
	return;
    }
    if (response.message === 'success') {
      var cache = response.payload;
      if( cache && cache.data && cache.updated && youngerThan(cache.updated,12)){
	SPACEWX['k_a'] = cache.data;	// use cached
	document.querySelector('#spacewx_container').querySelector('#k_index').innerHTML = SPACEWX['k_a'][1]; // Kp
	document.querySelector('#spacewx_container').querySelector('#k_index').setAttribute('title', SPACEWX['k_a'][0]); // time_tag
	document.querySelector('#spacewx_container').querySelector('#a_index').innerHTML = SPACEWX['k_a'][3]; // a_running
	document.querySelector('#spacewx_container').querySelector('#a_index').setAttribute('title', SPACEWX['k_a'][0]); // time_tag
	return;
      }
    }
    _fetch('https://services.swpc.noaa.gov/products/noaa-planetary-k-index.json') 
//		K+A:	[-1]->[1,3]
    .then(response => response.json())
    .then(data => {
	    SPACEWX['k_a'] = data.pop();
	    // ["time_tag","Kp","Kp_fraction","a_running","station_count"]
	    // ["2022-06-05 21:00:00.000","1","1.00","4","8"]
	    sendChromeMessage("set__SPACEWX_KA_cache",  {updated:Date.now(), data: SPACEWX['k_a']}  );
	    document.querySelector('#spacewx_container').querySelector('#k_index').innerHTML = SPACEWX['k_a'][1]; // Kp
	    document.querySelector('#spacewx_container').querySelector('#k_index').setAttribute('title', SPACEWX['k_a'][0]); // time_tag
	    document.querySelector('#spacewx_container').querySelector('#a_index').innerHTML = SPACEWX['k_a'][3]; // a_running
	    document.querySelector('#spacewx_container').querySelector('#a_index').setAttribute('title', SPACEWX['k_a'][0]); // time_tag
    }); // fetch
  });  // get__SPACEWX_KA_cache
  ////////
  chrome.runtime.sendMessage({ message: "get__ACTIVATIONS_cache" }, response => {
    if (response.message === 'success') {
      var cache = response.payload;
      if( cache && cache.data && cache.updated && youngerThan(cache.updated,20)){
	ACTIVATIONS = cache.data;	// use cached
	return;
      }
    }
    if( ! OPTIONS.enableNeededParks && ! OPTIONS.enableNeededLocations ){
	ACTIVATIONS = [];  // set to non-null so main handler thinks it is loaded. 
	return;
    }
    // refresh from pota.app API
    _fetch('https://api.pota.app/activation', POTA_API_OPTS)
    .then(response => response.json())
    .then(data => {
	    ACTIVATIONS = Array.from( data );
	    sendChromeMessage("set__ACTIVATIONS_cache",  {updated:Date.now(), data:ACTIVATIONS}  );
	    MAINTABLE.append(" "); // touch, to trigger initial update
/*
 [
 {
activator: "KN4ZMA"
activityEnd: null
activityStart: null
comments: "In BWCA, 5W CW, hoping to activate once or twice."
endDate: "2022-07-11"
endTime: "23:00"
frequencies: "7040"
locationDesc: "US-MN"
name: "Superior National Forest"
reference: "K-4491"
scheduledActivitiesId: 40958
schedulerUserId: 8938
startDate: "2022-07-02"
startTime: "01:00"
  },
  ...
  ]
*/
    }); // fetch
  });  // get__ACTIVATIONS_cache
  ////////
  ////////
} // if MYCALL

function setBackgroundForRow(btn, bgcolor) {
	if( ! btn ){
		return;
	}
	   var outer = LISTMODE ? btn.closest('tr') : btn.closest('div.v-card.v-sheet').querySelector('div.v-card__title.title');
	   if(outer){
		outer.style.setProperty( 'background-color', bgcolor, (bgcolor?'important':'') );
	   }
}

function setBackgroundForCallsign(callsign, bgcolor) {
	if(!callsign){
		return;
	}
	var search = '[data-callsign="' + callsign + '"]';
	var btn = document.querySelector(search);
	if(btn){
//	   btn.style.backgroundColor = bgcolor;
	   var outer = LISTMODE ? btn.closest('tr') : btn.closest('div.v-card.v-sheet').querySelector('div.v-card__title.title');
	   if(outer){
		outer.style.setProperty( 'background-color', bgcolor, (bgcolor?'important':'') );
		var btn2 = outer.querySelector('[data-marker="QSL_BTN"]');
		if(btn2){
//			btn2.style.setProperty('background-color', bgcolor, 'important');
		}
	   }
	}
}

function setCurrentCallsignBackground(bgcolor) {
    chrome.runtime.sendMessage({ 
        message: "get__selectedcall"
      }, response => {
      if (response.message === 'success') {
	var callsign = response.payload;
	setBackgroundForCallsign(callsign, bgcolor);
      }
    });
}

var selectedcall = '';  // TODO use ARRAY
var selectedcall2 = '';
var selectedcall3 = '';

chrome.runtime.sendMessage({ 
        message: "get__selectedcall"
      }, response => {
      if (response.message === 'success') {
	selectedcall = response.payload;
      }
});

var scan_call = '0E0';
var scanning = 0;
var SCANIDS = [];
function startScan(delta){
//  stopScan();
  var scanId = Math.random();
  scanning = scanId;
//  SCANIDS[scanId] = true;
  return scanNext(delta, scanId);
}
function stopScan(){
  scanning = 0;
//	SCANIDS = [];
//      Object.keys( SCANIDS ).forEach( x => { SCANIDS[x] = false } );
}
function scanNext(delta, scanId){   // delta=-1 for scan up, or +1 (default) for scan down
	if( scanning != scanId ){
		return;
	}
	delta ||= 1;
        var btns = Array.from( document.querySelectorAll('input#btn_qsy') );
	var i = btns.findIndex( x => x.getAttribute('data-callsign') === scan_call );
	i = (i+delta + btns.length) % btns.length;
	if(!btns[i]){return;}
	scan_call = btns[i].getAttribute('data-callsign');
	var spotKey = btns[i].getAttribute('data-spotKey');
	if( TOGGLES[spotKey] == 2 ){
		// skip if spot is toggled ~ to "negative"
		return scanNext(delta, scanId);
	}
	var p = btns[i];
	while( p = p.parentElement ){
	  if( p.style.display == 'none' ){
		// skip if spot is hidden by extra hide-spots filtering
		return scanNext(delta, scanId);
	  }
	}
	//console.log('scan qsy to ' + scan_call);
	btns[i].click();
	setTimeout(() => { scanNext(delta, scanId); }, document.getElementById('scan_dwell').value );
}


    var scanHTML = `
	&nbsp;
	<span id="scan_container"
	  class="ma-2 v-btn v-btn--is-elevated v-btn--has-bg theme--light v-size--default secondary"
	  style="
		display:none;
		__background-color: greenyellow;
		border-radius: 5px;
		padding: 6px;
		border: 1px solid grey;
		white-space: nowrap;
		"
	  >
      <input id="btn_scan" type=button value="SCAN" />
	<select id="scan_dwell" style="
		padding: 3px;
		padding-bottom: 0px;
		margin-left: 3px;
		background-color: lightgray;
		border-radius: 5px;
		border: 1px solid gray;
		__appearance: auto;
		font-size: smaller;
	    ">
	<option value="3000">3s</option>
	<option value="5000" SELECTED>5s</option>
	<option value="8000">8s</option>
	<option value="13000">13s</option>
	<option value="21000">21s</option>
	<option value="34000">34s</option>
	<option value="55000">55s</option>
	</select>
	</span>
    `;
    var scanNode = document.createElement('span');
    scanNode.innerHTML= scanHTML;
    document.querySelector('#upgrade_link').parentElement.appendChild(scanNode);

      var myAwardsNode = document.createElement('div');
      myAwardsNode.setAttribute('id','myawards_container');
      myAwardsNode.classList.add("col-12", "col");
      myAwardsNode.style.display	= "none";
      myAwardsNode.style.border		= "2px solid";
      myAwardsNode.style.borderRadius	= "5px";
      myAwardsNode.style.maxWidth	= "fit-content";
      myAwardsNode.style.padding	= "5px";
      myAwardsNode.style.marginBottom	= "5px";
      myAwardsNode.innerHTML = `
	Parks Hunted: <span id="parks_hunted"></span> / <span id="parks_next"></span>
	&nbsp;&nbsp;&nbsp;
	Late Shifts Hunted: <span id="lateshift_hunted"></span> / <span id="lateshift_next"></span>
	<br/>
      `;
      document.querySelector('#upgrade_link').parentElement.parentElement.parentElement.appendChild(myAwardsNode);

      var wxWidgetNode = document.createElement('div');
      wxWidgetNode.setAttribute('id','widget_rigref_container');
      wxWidgetNode.classList.add("col-12", "col");
      wxWidgetNode.style.display	= "none";
      wxWidgetNode.style.border		= "2px solid";
      wxWidgetNode.style.borderRadius	= "5px";
      wxWidgetNode.style.maxWidth	= "fit-content";
      wxWidgetNode.style.padding	= "5px";
      wxWidgetNode.style.marginBottom	= "5px";
      wxWidgetNode.style.marginLeft	= "10px";
      wxWidgetNode.innerHTML = `
	<a href="https://rigreference.com/solar" target="_blank"><img
		src="https://rigreference.com/solar/img/wide"
		border="0"
		style="height: 30px; vertical-align: middle;"
		onmouseover="this.style.height = '150px'"
		x__onmouseout="this.style.height = '30px'"
		onmouseout=" setTimeout(() => { this.style.height = '30px'; }, 1000 ); "
	></a>
      `;
      document.querySelector('#upgrade_link').parentElement.parentElement.parentElement.appendChild(wxWidgetNode);

      var spaceWxNode = document.createElement('div');
      spaceWxNode.setAttribute('id','spacewx_container');
      spaceWxNode.classList.add("col-12", "col");
      spaceWxNode.style.display	= "none";
      spaceWxNode.style.border		= "2px solid";
      spaceWxNode.style.borderRadius	= "5px";
      spaceWxNode.style.maxWidth	= "fit-content";
      spaceWxNode.style.padding	= "5px";
      spaceWxNode.style.marginBottom	= "5px";
      spaceWxNode.style.marginLeft	= "10px";
      spaceWxNode.innerHTML = `
	SFI=<span id="sfi"></span>
	&nbsp;
	A=<span id="a_index"></span>
	&nbsp;
	K=<span id="k_index"></span>
      `;
      document.querySelector('#upgrade_link').parentElement.parentElement.parentElement.appendChild(spaceWxNode);

      var greyLineNode = document.createElement('div');
      greyLineNode.setAttribute('id','greyline_container');
      greyLineNode.classList.add("col-12", "col");
      greyLineNode.style.display	= "none";
      greyLineNode.style.borderRadius	= "5px";
      greyLineNode.style.maxWidth	= "fit-content";
      greyLineNode.style.padding	= "0px";
      greyLineNode.style.marginLeft	= "10px";
      greyLineNode.innerHTML = `
	<a target="_blank" href="https://www.timeanddate.com/worldclock/sunearth.html"><img style="height: 40px" src="https://www.timeanddate.com/scripts/sunmap.php" /></a>
      `;
      document.querySelector('#upgrade_link').parentElement.parentElement.parentElement.appendChild(greyLineNode);

      var awardsEarnedNode = document.createElement('span');
      awardsEarnedNode.setAttribute('id','earned_container');
      awardsEarnedNode.classList.add("col-12", "col");
      awardsEarnedNode.style.display	= "none";
      awardsEarnedNode.style.border		= "2px solid";
      awardsEarnedNode.style.borderRadius	= "5px";
      awardsEarnedNode.style.maxWidth	= "fit-content";
      awardsEarnedNode.style.padding	= "5px";
      awardsEarnedNode.style.marginBottom	= "5px";
      awardsEarnedNode.innerHTML = '';
      document.querySelector('#upgrade_link').parentElement.parentElement.parentElement.appendChild(awardsEarnedNode);


    var bioNode = document.createElement('div');
    bioNode.setAttribute('id','bio_container_');
    bioNode.classList.add("col-3", "col");
//    bioNode.innerHTML = `<img id="bio_container_img" style="float:left"><textarea id="bio_container" style="display:none; height:100px; font-size: smaller; font-family: monospace; margin-left:10px; white-space:nowrap; width:100ch"></textarea>`;
//    document.querySelector('#upgrade_link').parentElement.parentElement.parentElement.appendChild(bioNode);

    var parkMapNode = document.createElement('div');
    parkMapNode.setAttribute('id','parkmap_container_');
    parkMapNode.classList.add("col-3", "col");

    var parkBioNode = document.createElement('div');
    parkBioNode.setAttribute('id','parkbio_container_');
    parkBioNode.classList.add("col-3", "col");
//    document.querySelector('#upgrade_link').parentElement.parentElement.parentElement.appendChild(parkBioNode);

    var spotBioNode = document.createElement('div');
    spotBioNode.setAttribute('id','spotbio_container_');
    spotBioNode.classList.add("col-3", "col");

    var activationBioNode = document.createElement('div');
    activationBioNode.setAttribute('id','activationbio_container_');
    activationBioNode.classList.add("col-12", "col");

    var infoPanesNode = document.createElement('div');
//    infoPanesNode.classList.add("row");
    infoPanesNode.style.width = "100%";
    //
    infoPanesNode.appendChild(bioNode);
    infoPanesNode.appendChild(parkBioNode);
    infoPanesNode.appendChild(parkMapNode);
    infoPanesNode.appendChild(spotBioNode);
    //
    infoPanesNode.appendChild(activationBioNode);
    //
    infoPanesNode.innerHTML = `
	    <div class="row">
	    	<div class="col col-3" id="bio_container_title"></div>
	    	<div class="col col-9" id="parkbio_container_title"></div>
	    </div>
	    <div class="row" style="margin-top: -10px">
	    	<div class="col col-3 col-md-3 col-sm-6 col-xs-12" id="bio_container_"></div>
	    	<div class="col col-3 col-md-3 col-sm-6 col-xs-12" id="parkbio_container_"></div>
	    	<div class="col col-3 col-md-3 col-sm-6 col-xs-12" id="parkmap_container_"></div>
	    	<div class="col col-3 col-md-3 col-sm-6 col-xs-12" id="spotbio_container_"></div>
	    </div>
	    <div class="row">
	    	<div class="col col-12" id="activationbio_container_"></div>
	    </div>
    `;
    document.querySelector('#upgrade_link').parentElement.parentElement.parentElement.appendChild(infoPanesNode);

    var logNode = document.createElement('div');
    logNode.setAttribute('id','log_container_');
    logNode.classList.add("col-12", "col");
    logNode.innerHTML = `<textarea id="log_container" style="display:none; width:100%; height:100%; font-size: smaller; font-family: monospace;"></textarea>`;
//    document.querySelector('#upgrade_link').parentElement.parentElement.parentElement.appendChild(logNode);
    logNode.style.width="100%";
    logNode.style.height="300px";
    logNode.style.border="1px solid black";
    MAINTABLE.parentElement.parentElement.append(logNode);


    var filterOptionsNode = document.createElement('div');
    filterOptionsNode.setAttribute('id','filter_container');
    filterOptionsNode.classList.add("col-12", "col");
    document.querySelector('#upgrade_link').parentElement.parentElement.parentElement.appendChild(filterOptionsNode);

    document.getElementById('btn_scan').addEventListener('click', (e) => {
	var x = document.getElementById('scan_container');
	// x=e.target
	if(scanning){
//		console.log('SCANNING OFF');
		//x.style.backgroundColor = 'greenyellow';
		x.classList.remove("error");
		x.classList.add("secondary");
		scanning = 0;
	}else{
//		console.log('SCANNING ON');
		//x.style.backgroundColor = 'red';
		x.classList.remove("secondary");
		x.classList.add("error");
		startScan(); // scanning = 1; scanNext();
	}
    });

function _updateBandsWorked(page, park_num, hunted){
	    var size = 100; // seems like size=100 is the max
	    var url = `https://api.pota.app/user/logbook?hunterOnly=1&page=${page}&size=${size}&reference=${park_num}`;
	    return _fetch(url, POTA_API_OPTS)
	    .then(response => response.json())
	    .then(data => {
		if( ! (data && data.entries && data.entries.length) ){
			return;
		}
		Array.from( data.entries ).forEach( (s) => {
		  if( s.reference == park_num ){ // sanity-check against false-positive search matches
			hunted["bandsWorked"][ s.band.toUpperCase() ] = s.qsoId; // store just the id, not whole object, to save space in extension storage cache.
		  }
		});
		hunted = _clean_N1CC_HUNTED_item(hunted);
		var idx = N1CC_HUNTED.findIndex( (element) => (element.reference == hunted.reference ) );
		N1CC_HUNTED[idx] = hunted;
		// data = {"count": 7, "entries": [{ "band": "40M", "mode": "PHONE", "reference": "K-6532", "loggedMode": "SSB", ...},   ... ] }
		if( data.entries.length && data.count > page*size && ! hunted["cacheComplete"] ){
		  return _updateBandsWorked(page+1, park_num, hunted);
		}
		return hunted;
	    });
}
/*
function updateBandsWorked(park_num, hunted, band, objs, badge){
  hunted.bandsWorked    ||= {};
  hunted.bandsWorkedStr ||= '';
  if( Object.keys(hunted.bandsWorked).length == hunted.bands ){
	return;
  }
  return _updateBandsWorked( 1, park_num, hunted ).then( x => {
    if( band ){
	if( ! hunted.bandsWorked[band] ){
	  if( ! objs.badges.innerHTML.match(/N1CC_HUNTED/) ){
	    badge.setAttribute('title', `${band} - Needed band for N1CC (${hunted.bandsWorkedStr})`);
	    objs.badges.append(badge);
	  }
	}
    }
  });
}
*/

function createDiv( classes ){
  var _e = document.createElement('div');
  Array.from( classes ).forEach( (c) => {
    _e.classList.add(c);
  } );
  return _e;
}
function buildInputDiv(xs,sm){
  var div = createDiv(['flex', xs||'xs12', sm||'sm6'])
	.appendChild( createDiv( ["v-input","v-input--is-label-active","v-input--is-dirty",THEMECLASS,"v-text-field","v-text-field--is-booted"] ) )
	.appendChild( createDiv(["v-input__control"]) )
	.appendChild( createDiv(["v-input__slot"]) )
	.appendChild( createDiv(["v-text-field__slot"]) )
	;
  return div;
}

function RST_ctrlNum(event) {
  if (event.ctrlKey && event.key.match(/^[0-9]$/) ){
    event.target.value = event.target.getAttribute('data-defaultrst').replace(/(.).(.?)/, "$1" + event.key + "$2");
    event.preventDefault();
  }
}

function toADIF(k,v){
  return "<"+k+":"+v.length+">"+v;
}

function submitLogEntry( fetch_url ){
//  console.log(" FETCH " , fetch_url);
  if( OPTIONS.loggingConfig.method == 'qrz' ){
	// we expect here that fetch_url is just a query string
	let searchParams = new URLSearchParams(fetch_url);
	var adif = Array.from( searchParams.entries() ).map( entry => toADIF(entry[0], entry[1]) ).join('')
	  + '<eor>';
//	  console.log("ADIF:", adif);
	// https://www.qrz.com/docs/logbook/QRZLogbookAPI.html
	// https://logbook.qrz.com/apicheck
	_fetch( 'https://logbook.qrz.com/api', {
		method: 'POST',
		mode: 'no-cors',
		headers: {
			'Content-Type': 'application/x-www-form-urlencoded',
		},
		body: "key="+OPTIONS.loggingConfig.target_apikey
			+ "&ACTION=INSERT"
			+ "&ADIF="+encodeURIComponent(adif)
	} )
	  // TODO -- this does not seem to get the response ... expecting something like "LOGID=733604667&COUNT=1&RESULT=OK"  or an error message.
	  .then(response => {console.log("BODY: ",response.body); return response.text() })
	  .then( function(text){ console.log("QRZ.com RESPONSE:"); console.log(text) } )
	  .catch(err => console.log("QRZ.com ERROR:", err))
	;

  }else{ // straight GET
	_fetch( fetch_url, { // mode: 'no-cors'
		} );
  }
}

function logBtnClick(e2, doReSpot){
	    var _url = e2.target.getAttribute('data-URL');
	    var call = e2.target.getAttribute('data-call');
	    var ping_url = e2.target.getAttribute('data-ping-URL');
	    // hack to trap the known error, so it does not create errant log entries.
	    var spotCall = Array.from( e2.target.closest(".v-card").querySelectorAll('label') ).find( el => el.textContent.match('Activator Callsign') ).nextSibling.value;
	    if( spotCall != call ){
		alert(`ERROR -- Call mismatch [${spotCall} vs ${call}] -- refresh page before logging`);
		return;
	    }
	    if( ping_url ){
	setTimeout(() => {
	  fetch( ping_url, {} )
		.then(function(response) {
		  if ( !response.ok // || ! data.success
			){
			var s = `PING FAILED [${response.status}] ${response.statusText}`;
			_log(ping_url);
			_log(s);
			_log( JSON.stringify(response) );
			console.log(ping_url);
			console.log(s);
			console.log( JSON.stringify(response) );
			alert("ERROR -- QSO NOT LOGGED");
			var msg = "VERIFY THAT ham-apps-proxy.exe AND your logging app are RUNNING";
			document.querySelector('#label_update').innerHTML = `ERROR -- ${msg}`;
			document.querySelector('#label_update').style.display = '';
		  }
		  return response;
		});
	}, 1000 ); //setTimeout
	    }
	    //
	    var park_num = e2.target.getAttribute('data-park_num');
	    var park_region = e2.target.getAttribute('data-park_region');
	    var RST_SENT = e2.target.closest(".v-card").querySelector("#RST_SENT").value;
	    var RST_RCVD = e2.target.closest(".v-card").querySelector("#RST_RCVD").value;
	    var OTHER_OPS = e2.target.closest(".v-card").querySelector("#OTHER_OPS").value.split(/[^A-Z0-9\/]+/i).filter(s=>s).map(s=>s.toUpperCase().trim());
	    var OTHER_PARKS = [
		e2.target.closest(".v-card").querySelector("#PARKNUM2").value,
		e2.target.closest(".v-card").querySelector("#PARKNUM3").value,
		e2.target.closest(".v-card").querySelector("#PARKNUM4").value,
		e2.target.closest(".v-card").querySelector("#PARKNUM5").value,
	    ].filter(s=>s).map(s=>s.toUpperCase().trim());
//	    var secondHunter = e2.target.closest(".v-card").querySelector("#SECONDHUNTER").value;
	    var ALL_PARKS = [park_num].concat(OTHER_PARKS);
	    var alsoOtherParks = OTHER_PARKS.length ? '{Also: ' + OTHER_PARKS.join(',') + '} ' : '';
	    var RX_PWR = e2.target.closest(".v-card").querySelector("#RX_PWR").value;
	    var activatorPwr = RX_PWR ? '{Pwr: ' + RX_PWR + 'W} ' : '';
	    TX_PWR = e2.target.closest(".v-card").querySelector("#TX_PWR").value; // global
	    sendChromeMessage("set__TX_PWR",  TX_PWR  );
	    var info = PARK_INFO[park_num] && PARK_INFO[park_num]['info'] ? PARK_INFO[park_num]['info'] : {};
	    var now = new Date().toISOString().slice(11,13) + "00"; // HH00  in UTC
	    var shiftInfo =
			  hourBetween(now, info['lateshift_start' ], info['lateshift_end' ]) ? '{LateShift} '
			: hourBetween(now, info['earlyshift_start'], info['earlyshift_end']) ? '{EarlyShift} '
			: '';
	    var COMMENTS_obj = Array.from( e2.target.closest(".v-card").querySelectorAll("label") ).find(x => x.innerText=='Comments' ).closest('div').querySelector('input')
	    var COMMENTS = COMMENTS_obj.value;
// TODO    var p = PARK_INFO[park_num]["info"]; // grid6  locationDesc
	    _url = _url; // replace
	    _url = _url.replaceAll('__RST_RCVD__', RST_RCVD );
	    _url = _url.replaceAll('__RST_SENT__', RST_SENT );
	    _url = _url.replaceAll('__TX_PWR__', TX_PWR );
	    _url = _url.replaceAll('__RX_PWR__', RX_PWR );
	    _url = _url.replaceAll('__COMMENTS__', activatorPwr + alsoOtherParks + shiftInfo + COMMENTS );
	    // inject QSODATE/TIME_ON here (for loggers using ADIF format date/time), so it's log-click time, not open-dialog-time.
	    _url = _url.replaceAll('__DATE_YYYYMMDD__', new Date().toISOString().slice(0 ,10).replaceAll("-","") );
	    _url = _url.replaceAll('__TIME_HHMMSS__',   new Date().toISOString().slice(11,19).replaceAll(":","") );
	    _url = _url.replaceAll('__LOG4OM_PARKS__', JSON.stringify(
		    ALL_PARKS.map(function(n){
			    return {"AC":"POTA", "R":n, "G":park_region, "SUB":[], "GRA":[]}
		    }) ) );
	    _url = _url.replaceAll('__PARK_NUMS__', ALL_PARKS.join(',') );
	    var fetch_url = _url.replaceAll('__CALL__', call );
	    submitLogEntry( fetch_url );
//console.log(" FETCH " , fetch_url);
//	    fetch( fetch_url, { mode: 'no-cors' } );
	    OTHER_OPS.forEach( (op_call) => {
		fetch_url = _url.replaceAll('__CALL__', op_call );
	        submitLogEntry( fetch_url );
//console.log(" FETCH OTHER CALL " , fetch_url);
//		fetch( fetch_url, { mode: 'no-cors' } );
	    } );
	    if( doReSpot ){
		// prepend spot comments wthe report/homeQTH (e.g. "[59 ENY] ") then reset it after spot sent.
		COMMENTS_obj.value = ""
		    	+ '['
			+ RST_SENT
			+ (OPTIONS.loggingConfig.homeQTH ? " " + OPTIONS.loggingConfig.homeQTH : '')
			+ '] '
			+ activatorPwr
			+ alsoOtherParks
		    	+ (OTHER_OPS.length   ? '{With: ' + OTHER_OPS.join(',')   + '} ' : '')
			+ COMMENTS
		    ;
		COMMENTS_obj.dispatchEvent(new Event('input', {bubbles:true}));  // Vuetify ignores the value, unless it's from an input event, since bound to the Vue model/variable.
		Array.from( e2.target.closest(".v-card__actions").querySelectorAll("button") ).find(x => x.querySelector("span").innerText=='SPOT' ).click();
//		COMMENTS_obj.value = COMMENTS; // TODO -- timeout(100ms) ??
//		setTimeout(() => { COMMENTS_obj.value = COMMENTS; }, 200 );
	    }
	    Array.from( e2.target.closest(".v-card__actions").querySelectorAll("button") ).find(x => x.querySelector("span").innerText=='CANCEL' ).click();
	// Attempt (hack/brute-force) to fix elusive bug where QSL button gets "jumbled".
	Array.from( MAINTABLE.querySelectorAll('[data-marker="QSL_BTN"]') , e => e.remove() );
	Array.from( MAINTABLE.querySelectorAll('[data-marker="LOG_BTN"]') , e => e.remove() );
	Array.from( MAINTABLE.querySelectorAll('[data-marker="RST_RCVD"]') , e => e.remove() );
	Array.from( MAINTABLE.querySelectorAll('[data-marker="TOGGLE_BTN"]') , e => e.remove() );
	MAINTABLE.append(" "); // touch, to trigger initial update
	if( OPTIONS.loggingConfig.forceRefresh ){
		setTimeout(() => { document.location.reload(); }, 500 ); // short delay (in milliseconds) to give logging fetch's time to fire.
	}
}

function hijackSpotForm(url,call,park_num,park_region,mode, logBtn, ping_url ){
    var spotForm = Array.from(document.querySelectorAll('.v-form')).find(el => 
	el.closest(".v-dialog").style.display != 'none'
    );
    if( spotForm.querySelector(".layout.wrap").innerHTML.match(/RST_RCVD/) ){
	return;
    }
    spotForm.closest(".v-dialog").style.border = "4px solid blue";
    var RSTDefault = mode == "SSB"  ? "59"
		 : mode == "CW"   ? "599"
		 : mode == "RTTY" ? "599"
		 :                   ""
    ;
    var rstR = buildInputDiv();
    rstR.getRootNode().setAttribute('data-marker','RST_RCVD');
    rstR.innerHTML = `
	  <label class="v-label v-label--active ${THEMECLASS}" style="left: 0px; right: auto; position: absolute;">
		  RST Rcvd
		  <span style="font-size:smaller">(ctrl-# for 5#/5#9)</span>
		  </label>
  	<input type="text" id="RST_RCVD" value="${RSTDefault}" data-defaultrst="${RSTDefault}">
    `;
    rstR.addEventListener('keydown', (event) => RST_ctrlNum(event) );
   //
    var rstS = buildInputDiv();
    rstS.getRootNode().setAttribute('data-marker','RST_SENT');
    rstS.innerHTML = `
	  <label class="v-label v-label--active ${THEMECLASS}" style="left: 0px; right: auto; position: absolute;">
		  RST Sent
		  <span style="font-size:smaller">(ctrl-# for 5#/5#9)</span>
		  </label>
  	<input type="text" id="RST_SENT" value="${RSTDefault}" data-defaultrst="${RSTDefault}">
    `;
    rstS.addEventListener('keydown', (event) => RST_ctrlNum(event) );
   //
    var TBD = document.createElement('div');
    TBD.classList.add('flex', 'xs12', 'sm6');
    TBD.innerHTML = '&nbsp;';
   //
    var txPwr = buildInputDiv('xs12','sm3');
    txPwr.getRootNode().setAttribute('data-marker','TX_PWR');
    txPwr.innerHTML = `
	  <label class="v-label v-label--active ${THEMECLASS}" style="left: 0px; right: auto; position: absolute;">
		  TX PWR </label>
  	<input type="text" id="TX_PWR" value="${TX_PWR}">
    `;
   //
    var rxPwr = buildInputDiv('xs12','sm3');
    rxPwr.getRootNode().setAttribute('data-marker','RX_PWR');
    rxPwr.innerHTML = `
	  <label class="v-label v-label--active ${THEMECLASS}" style="left: 0px; right: auto; position: absolute;">
		  RX PWR </label>
  	<input type="text" id="RX_PWR" value="">
    `;
   //
    var addtlCalls = buildInputDiv('xs12','sm6');
    addtlCalls.getRootNode().setAttribute('data-marker','OTHER_OPS');
    addtlCalls.innerHTML = `
	  <label class="v-label v-label--active ${THEMECLASS}" style="left: 0px; right: auto; position: absolute;">
		  Other Ops
		  <span style="font-size:smaller">(comma/space-delim)</span>
		  </label>
  	<input type="text" id="OTHER_OPS" value="">
    `;
   //
    var parkNum2 = buildInputDiv('xs12','sm3');
    parkNum2.getRootNode().setAttribute('data-marker','PARKNUM2');
    parkNum2.innerHTML = `
	  <label class="v-label v-label--active ${THEMECLASS}" style="left: 0px; right: auto; position: absolute;">
		  Park Ref #2
		  </label>
  	<input type="text" id="PARKNUM2" value="">
    `;
   //
    var parkNum3 = buildInputDiv('xs12','sm3');
    parkNum3.getRootNode().setAttribute('data-marker','PARKNUM3');
    parkNum3.innerHTML = `
	  <label class="v-label v-label--active ${THEMECLASS}" style="left: 0px; right: auto; position: absolute;">
		  Park Ref #3
		  </label>
  	<input type="text" id="PARKNUM3" value="">
    `;
   //
    var parkNum4 = buildInputDiv('xs12','sm3');
    parkNum4.getRootNode().setAttribute('data-marker','PARKNUM4');
    parkNum4.innerHTML = `
	  <label class="v-label v-label--active ${THEMECLASS}" style="left: 0px; right: auto; position: absolute;">
		  Park Ref #4
		  </label>
  	<input type="text" id="PARKNUM4" value="">
    `;
   //
    var parkNum5 = buildInputDiv('xs12','sm3');
    parkNum5.getRootNode().setAttribute('data-marker','PARKNUM5');
    parkNum5.innerHTML = `
	  <label class="v-label v-label--active ${THEMECLASS}" style="left: 0px; right: auto; position: absolute;">
		  Park Ref #5
		  </label>
  	<input type="text" id="PARKNUM5" value="">
    `;
   //
//    var secondHunter = buildInputDiv('xs12','sm12');
//    secondHunter.getRootNode().setAttribute('data-marker','SECONDHUNTER');
//    secondHunter.innerHTML = `
//	  <label class="v-label v-label--active ${THEMECLASS}" style="left: 0px; right: auto; position: absolute;">
//		  Park Ref #5
//		  </label>
//	<select id="SECONDHUNTER">
//		<option value="">-</option>
//    `;
//    secondHunter.innerHTML += Object.keys(HUNTER_CALLS).filter(s=>s!=MYCALL).sort().map(s=>`<option value="${s}">${s}</option>`).join('');
//    secondHunter.innerHTML += `
//	</select>
//    `;
   //
    spotForm.querySelector(".layout.wrap").append( rstS.getRootNode() );
    spotForm.querySelector(".layout.wrap").append( rstR.getRootNode() );
    spotForm.querySelector(".layout.wrap").append( txPwr.getRootNode() );
    spotForm.querySelector(".layout.wrap").append( rxPwr.getRootNode() );
    spotForm.querySelector(".layout.wrap").append( addtlCalls.getRootNode() );
    spotForm.querySelector(".layout.wrap").append( parkNum2.getRootNode() );
    spotForm.querySelector(".layout.wrap").append( parkNum3.getRootNode() );
    spotForm.querySelector(".layout.wrap").append( parkNum4.getRootNode() );
    spotForm.querySelector(".layout.wrap").append( parkNum5.getRootNode() );
//    if( Object.keys(HUNTER_CALLS).length > 1 ){
//      spotForm.querySelector(".layout.wrap").append( secondHunter.getRootNode() );
//    }
    //spotForm.querySelector(".layout.wrap").append( TBD );
   //
    var spotCall = Array.from( spotForm.querySelectorAll('label') ).find( el => el.textContent.match('Activator Callsign') ).nextSibling.value;
   //
    logBtn.setAttribute('data-URL', url);
    logBtn.setAttribute('data-ping-URL', ping_url || ''); // make sure not a string of "undefined"
    logBtn.setAttribute('data-call', call);
    logBtn.setAttribute('data-park_num', park_num);
    logBtn.setAttribute('data-park_region', park_region);
    logBtn.value = 'Log' +' '+call;
    var logspotBtn = logBtn.cloneNode(true);
    logspotBtn.value = 'Spot+Log';
    if( call != spotCall ){
	    // hack to trap the known error, so it does not create errant log entries.
        spotForm.closest(".v-card").querySelector(".v-card__actions").append(`ERROR -- Call mismatch [${spotCall} vs ${call}] -- refresh page before logging`);
    }else{
        logBtn.addEventListener('click', (e2) => logBtnClick(e2, false) );
        logspotBtn.addEventListener('click', (e2) => logBtnClick(e2, true) );
        spotForm.closest(".v-card").querySelector(".v-card__actions").appendChild(logspotBtn);
        spotForm.closest(".v-card").querySelector(".v-card__actions").appendChild(logBtn);
    }
}

/////////////////////////////////////////////////////////////

var SPOTS_TO_HIDE = {};
chrome.runtime.sendMessage({ message: "get__SPOTS_TO_HIDE" }, response => {
      if (response.message === 'success') {
	      SPOTS_TO_HIDE = response.payload || {};
      }
});

function updateFilterHide(k,v){
	SPOTS_TO_HIDE[k] = v;
	sendChromeMessage("set__SPOTS_TO_HIDE",  SPOTS_TO_HIDE );
	MAINTABLE.append(" "); // touch, to trigger initial update
}

var SPOT_FILTER_DETAILS_TOGGLE = 0; 

function hourBetween(t, start, end){
	// all 3 params are UTC HHMM
	// return bool for:    start <= t < end
	if ( end < start ){
		end = ( (end*1) + 2400 ).toString().padStart(2, '0');
	}
	if ( t < start ){
		t = ( (t*1) + 2400 ).toString().padStart(2, '0');
	}
	var isBetween =  ( start <= t && t < end );
	// console.log( `==== hourBetween( ${t}, ${start}, ${end} ) => ${isBetween}` );
	return isBetween;
}
/////////////////////////////////////////////////////////////

var observer;
if( observer == null ){
observer = new MutationObserver(mutations => {
    if( document.location.hash == "#/activations" ){
	console.log("Switched to ACTIVATIONS page");
	document.location.reload();
	return;
    }
    if( document.location.hash.match("#/profile/.+") ){
	console.log("Switched to PROFILE page");
	document.location.reload();
	return;
    }
    if( document.location.hash != "#/" ){
	console.log("NO LONGER ON SPOTS PAGE");
	return;
    }

    if( OPTIONS == null
		|| WAS_HUNTED == null
		|| LOCATIONS_HUNTED == null
		|| PARKS_HUNTED == null
		|| N1CC_HUNTED == null
		|| OPS_HUNTED == null
//	    	|| ! LATESTVER
    	){
	    // skip while still loading/initializing the async-fetch of dependencies
	    console.log("LOADING:"
		, 'OPTIONS: '      + (OPTIONS      ? 'Y' : '-')
		, 'WAS_HUNTED: '   + (WAS_HUNTED   ? 'Y' : '-')
		, 'LOCATIONS_HUNTED: ' + (LOCATIONS_HUNTED   ? 'Y' : '-')
		, 'PARKS_HUNTED: ' + (PARKS_HUNTED ? 'Y' : '-')
		, 'N1CC_HUNTED: '  + (N1CC_HUNTED  ? 'Y' : '-')
		, 'OPS_HUNTED: '   + (OPS_HUNTED   ? 'Y' : '-')
		, 'LATESTVER: '    + (LATESTVER    ? 'Y' : '-')
	    );
	    return;
    }
    document.getElementById('btn_info').classList.remove('warning');

//console.log("UPDATE TRIGGERED");
    observer.disconnect(); // turn observer off;

    var SPOT_ATTRIBUTES = { "bands":{}, "modes":{}, "programs":{}, "regions":{} };

    if( OPTIONS.enableHideSpotterCall ){
      if( LISTMODE ){
	MAINTABLE.parentElement.tHead.rows[0].cells[5].style.display = 'none';
	for (let row of MAINTABLE.rows) {
	      if( row.cells[5] && row.cells[5].style ){
	        row.cells[5].style.display = 'none';
	      }
	}
      }
      if( CARDMODE ){
	var rows = Array.from( MAINTABLE.querySelectorAll('div.v-card.v-sheet') );
	for (let row of rows) {
	    var cells = Array.from( row.querySelectorAll("div.v-list-item") );
	    if( cells[3] ){
		cells[3].style.display = 'none';
	    }
	}
      }
    }
    if( OPTIONS.enableHideSpotterComment ){
      if( LISTMODE ){
	MAINTABLE.parentElement.tHead.rows[0].cells[6].style.display = 'none';
	for (let row of MAINTABLE.rows) {
	      if( row.cells[6] && row.cells[6].style ){
	        row.cells[6].style.display = 'none';
	      }
	}
      }
      if( CARDMODE ){
	var rows = Array.from( MAINTABLE.querySelectorAll('div.v-card.v-sheet') );
	for (let row of rows) {
	    var cells = Array.from( row.querySelectorAll("div.v-list-item") );
	    if( cells[4] ){
		cells[4].style.display = 'none';
	    }
	}
      }
    }

    var rows;
    if( LISTMODE ){
	rows = MAINTABLE.rows;
	// prevent the "jumbling" issue ... clear any added elements,
	// so they can be re-made. otherwise they persist in a fixed row,
	// even if spots are re-ordered.
        Array.from( MAINTABLE.querySelectorAll('[data-marker]:not([data-marker=""])') , e => {
			e.remove();
        });
    }
    if( CARDMODE ){
	rows = Array.from( MAINTABLE.querySelectorAll('div.v-card.v-sheet') );
    }
    var COLOR1 = DARKMODE ? 'darkslategrey'	: 'lightgreen'; // 'yellow';
    var COLOR2 = DARKMODE ? 'dimgrey'		: 'inherit'; // 'lightyellow'
    var COLOR3 = DARKMODE ? 'black'		: 'inherit'; // 'lightgoldenrodyellow';
    for (let row of rows) {
      var cells = LISTMODE ? row.cells : Array.from( row.querySelectorAll("div.v-list-item") );
      var objs = {
	call: (LISTMODE ? cells[1]
		: ( row.querySelector("div.v-card__title.title a")
			||  row.querySelector("div.v-card__title.title span span") // User Stats mode
		) ),
	freq:		(LISTMODE ? cells[2] : cells[2] ),
	park_num:	(LISTMODE ? cells[3] : row.querySelector("div.v-card__title.title span") ),
	park_name:	(LISTMODE ? cells[3] : cells[0] ),
	park_region:	(LISTMODE ? cells[4] : cells[1] ),
	btns:		(LISTMODE ? cells[8] : row.querySelector("div.v-card__actions") ),
	extLinks:	(LISTMODE ? cells[7] : row.querySelector(".v-btn") ),
	historyBtn:	(LISTMODE ? cells[7].querySelector(".v-btn") : null ),
	spottime:	(LISTMODE ? cells[0] : cells[5] ),
      };
      if(CARDMODE && objs.btns){
	      objs.historyBtn = objs.btns.querySelector(".v-btn--outlined");
      }
      if( !cells.length ) {
	    continue;
      }
      var btnStyle = "border: 2px solid grey; border-radius: 5px; padding:3px;";
///////////////////////////////////////
      if( !objs.call ) {
            console.log("ERROR -- no objs.call", objs);
	    continue;
      }
      var call = objs.call.innerText.trim();
      var freqOrig;
      if( objs.freq.querySelector('input#btn_qsy') ){
	freqOrig = objs.freq.querySelector('input#btn_qsy').getAttribute('data-freqorig');
      }else{
        freqOrig = objs.freq.innerText.trim();
      }
      var freq = freqOrig;
      var freqLabel = freq.replace(/(\d)(\d\d\d)($| |\D)/,'$1.$2$3');
      var park_num = objs.park_num.innerText.trim();
      var park_name = objs.park_name.innerText.trim().replaceAll('"',"'"); // replace double quotes with single quote
	    	// todo escape a "&"
      park_name = park_name.replace(/^\S+\s+/, ''); // strip park number of of front
      var park_region = objs.park_region.innerText.replace(/\n.+/,'').trim();
      park_region = park_region.replace(/(0|\d+\/\d+)$/,'').trim(); // CARDMODE hack
      var park_program = park_region.replace(/-.+/,'');
      if( LISTMODE ){
	      park_num = park_num.replace(/\s.+/, '');
      }
      if( CARDMODE ){
	      park_num = park_num.replace(/.+ @\s*/, '');
	      park_name = park_name.replace(/^[^,]+,\s*/, '');
      }
      var modeLog = freq.match(/\((.+)\)/);
      modeLog = modeLog != null ? modeLog[1] : 'SSB';
      freq = freq.replace(/ .+/,''); // freq is now numeric, in kHz
      var modeQsy =
		  modeLog.match( /FT.*/    ) ? OPTIONS.qsyConfig.dataMode
		: modeLog.match( /JT.*/    ) ? OPTIONS.qsyConfig.dataMode
		: modeLog.match( /.*PSK.*/ ) ? OPTIONS.qsyConfig.dataMode
		: modeLog.match( /SSTV/    ) ? OPTIONS.qsyConfig.dataMode
		: modeLog.match( /CW/      ) ? OPTIONS.qsyConfig.cwMode
		: modeLog.match( /SSB/ ) && freq ? ( freq >= 10000 || (freq>=5000 && freq<6000) ? 'USB' : 'LSB' ) // USB for 30m & up, plus 60M; LSB for 40M, 80m & below
		:                              modeLog  // SSB, AM, FM, CW, etc
      ;
      var band= freq >=10500000 ? '' // unknown
	    // http://www.arrl.org/band-plan
	    :	freq >=10000000 ?  '3cm'
	    :	freq >= 5650000 ?  '5cm'
	    :	freq >= 3300000 ?    '' // ??
	    :	freq >= 2300000 ? '13cm'
	    :	freq >= 1240000 ? '23cm'
	    :	freq >=  902000 ? '33cm'
	    :	freq >=  420000 ? '70cm'
	    :	freq >=  222000 ?'1.25M'
	    :	freq >=  140000 ?   '2M'	//144000
	    :	freq >=   50000 ?   '6M'
	    :	freq >=   28000 ?  '10M'
	    :	freq >=   24000 ?  '12M'
	    :	freq >=   21000 ?  '15M'
	    :	freq >=   18000 ?  '17M'
	    :	freq >=   14000 ?  '20M'
	    :	freq >=   10000 ?  '30M'
	    :	freq >=    7000 ?  '40M'
	    :	freq >=    5000 ?  '60M'
	    :	freq >=    3500 ?  '80M'
	    :	freq >=    1800 ? '160M'
	    :			'' // unknown
      ;
      var isUSGeneralOutOfBand = false
	    || (  3500 <= freq && freq <=  3525 )
	    || (  3600 <= freq && freq <=  3800 )
	    || (  7000 <= freq && freq <=  7025 )
	    || (  7125 <= freq && freq <=  7175 )
	    || ( 14000 <= freq && freq <  14025 )
	    || ( 14150 <= freq && freq <  14225 )
	    || ( 21000 <= freq && freq <  21025 )
	    || ( 21200 <= freq && freq <  21275 )
      ;
      var isSSBUSBandEdges = false
	    || [1800, 3600, 3700, 3800, 7125, 7175]
		.filter((x) => { var y = freq - x; return 0<=y && y<3 } ).length > 0
	    || [14350, 18168, 21450, 24990, 29700, 54000]
		.filter((x) => { var y = x - freq; return 0<=y && y<3 } ).length > 0
      ;

      var today = new Date();
      var todayStr = today.getUTCFullYear() + '-' + (1+today.getUTCMonth()).toString().padStart(2, '0') + '-' + today.getUTCDate();
      var spotKey = [ call, park_num, band, modeLog, todayStr ].join(":");

    if( OPTIONS.enableSpotTimeAbbrv && LISTMODE ){
	var timeObj = Array.from(objs.spottime.querySelectorAll('span')).find(el => el.textContent.match(' ago'));
	if( timeObj){
		var s = timeObj.innerText
				.replace(/ mins? ago/, ' ')
				.replace(/\d+ sec ago/, '<1 ')
		;
		//timeObj.innerHTML = s;
//		timeObj.prepend(s);
		var time2 = timeObj.parentElement.querySelector('span#time_label');
		if( ! time2 ){
		  time2 = document.createElement('span');
		  time2.setAttribute('id', 'time_label');
		  timeObj.parentElement.append(time2);
		}
		// TODO -- don't make this dependent upon enableSpotTimeAbbrv option.
		var info = PARK_INFO[park_num] ? PARK_INFO[park_num]['info'] : {};
		if( info['longitude'] ){
		  var now = new Date().toISOString().slice(11,13) + "00"; // HH00  in UTC
		  if( hourBetween(now, info['lateshift_start'], info['lateshift_end']) ){
			  s += ` =LS>${info['lateshift_start']}`;
		  }else if( hourBetween(now, info['earlyshift_start'], info['earlyshift_end']) ){
			  s += ` =ES<${info['earlyshift_end']}`;
		  }
		}else{
		  s += '.';
		}
		time2.innerHTML = s;
		timeObj.style.display = 'none';
	}
//      objs.spottime.innerHTML = objs.spottime.innerHTML.toString().replace(/ mins? ago/, ' ');
//      objs.spottime.innerHTML = objs.spottime.innerHTML.toString().replace(/\d+ sec ago/, '<1 ');
    }


    SPOT_ATTRIBUTES['bands'   ][band]         ||= {};
    SPOT_ATTRIBUTES['modes'   ][modeLog]      ||= {};
    SPOT_ATTRIBUTES['programs'][park_program] ||= {};
    SPOT_ATTRIBUTES['regions' ][park_region]  ||= {};
    // var spotSlot = band + '/' + modeLog + '/' + park_region;
    SPOT_ATTRIBUTES['bands'   ][band        ][modeLog + '/' + park_region]++;
    SPOT_ATTRIBUTES['modes'   ][modeLog     ][band + '/' + park_region   ]++;
    SPOT_ATTRIBUTES['programs'][park_program][band + '/' + modeLog + '/' + park_region]++;
    SPOT_ATTRIBUTES['regions' ][park_region ][band + '/' + modeLog       ]++;

    var hideSpot = OPTIONS.enableSpotFilters && (
	       false
	    || SPOTS_TO_HIDE[band]
	    || SPOTS_TO_HIDE[modeLog]
	    || SPOTS_TO_HIDE[park_program]
	    || SPOTS_TO_HIDE[park_region]
    	) ? true : false;
    var outer = LISTMODE ? row : row.parentElement;
    outer.style.display = hideSpot ? 'none' : '';


    if( ! objs.park_num.innerHTML.match(/BADGE_HOLDER/) ){
	  objs.badges = document.createElement('div');
	  objs.badges.setAttribute('data-marker','BADGE_HOLDER');
	  objs.badges.classList.add('v-badge__wrapper');
	  objs.badges.style.position='absolute';
	  if( LISTMODE ){
	    objs.badges.style.top='30px';
	    objs.badges.style.left='20px';
	  }
	  if( CARDMODE ){
	    objs.badges.style.display = 'contents';
	  }
	  objs.badges.style.fontFamily='monospace';
	  objs.badges.innerHTML = '';
	  objs.park_num.append(objs.badges);
    }else{
	  objs.badges = objs.park_num.querySelector('[data-marker="BADGE_HOLDER"]');
    }

//-------------------------------------
    if( OPTIONS.enableNeededLocations && ! objs.park_region.innerHTML.match(/LOCATIONS_HUNTED/) ){
      var loc = park_region.replace(/,.+/,'');
      if(LOCATIONS_HUNTED.length ){
        var parkWorked = PARKS_HUNTED.find( x => x.reference === park_num );
        var hunted = LOCATIONS_HUNTED.find( x => x.locationDesc === loc );
		// keys: locationDesc, locationId, locationName, numParksActivated, numParksAvailable, toGo
//if Region not found as a Location, then "0" warning badge;  else if new-park then accent badge with the numParksActivated / numParksAvailable count.
	var label = hunted == null ? "0"
	      	: hunted.numParksActivated + "/" + hunted.numParksAvailable
	      ;
	var title = hunted == null ? "New needed location"
	      	: hunted.numParksActivated + "/" + hunted.numParksAvailable + " references worked for this location"
	      ;
        if( ! parkWorked ){ // new park#
//console.log('NEEDED', park_region, park_num, call, hunted);
	  var badge = document.createElement('span');
	  badge.setAttribute('data-marker','LOCATIONS_HUNTED');
	  badge.classList.add('v-badge__wrapper');
	  if( LISTMODE ){
	    badge.style.position='relative';
	    badge.style.top='15px';
	    badge.style.left='-15px';
	  }else{
	    badge.style.position='absolute';
	    badge.style.top='0px';
	    badge.style.left='35px';
	  }
	  badge.style.fontFamily='monospace';
	  badge.innerHTML = `<span class="v-badge__badge warning" title="${title}">${label}</span>`;
	  objs.park_region.append(badge);
        }
      }
    } // OPTIONS.enableNeededLocations

//-------------------------------------
    if( OPTIONS.enableNeededWAS && ! objs.park_region.innerHTML.match(/WAS_HUNTED/) ){
      var state = park_region.replace(/,.+/,'');
      if(WAS_HUNTED.length && state.match(/^US-..$/) && state != 'US-VI' ){
        var hunted = WAS_HUNTED.find( x => x.locationDesc === state );
        if( hunted == null ){
//console.log('NEEDED', park_region, park_num, call, hunted);
	  var badge = document.createElement('span');
	  badge.setAttribute('data-marker','WAS_HUNTED');
	  badge.classList.add('v-badge__wrapper');
	  if( LISTMODE ){
	    badge.style.position='relative';
	    badge.style.top='10px';
	    badge.style.left='-5px';
	  }else{
	    badge.style.position='absolute';
	    badge.style.top='0px';
	    badge.style.left='35px';
	  }
	  badge.style.fontFamily='monospace';
	  badge.innerHTML = `<span class="v-badge__badge warning">WAS</span>`;
	  objs.park_region.append(badge);
        }
      }
    } // OPTIONS.enableNeededWAS

//-------------------------------------
    if( OPTIONS.enableNeededParks && ! objs.park_num.innerHTML.match(/PARKS_HUNTED/) ){
      if(PARKS_HUNTED.length){
        var hunted = PARKS_HUNTED.find( x => x.reference === park_num );
	var badge = document.createElement('span');
	badge.setAttribute('data-marker','PARKS_HUNTED');
	badge.classList.add('v-badge__badge');
	badge.style.position = 'relative';
	badge.style.marginRight = '5px';
        if( hunted != null ){
	  badge.classList.add(
		  hunted.qsos % 20 >= 17	? 'warning'  // Repeat Offender Hunter levels are every 20, so warn when close
		: hunted.qsos >= 20		? 'grey'
		:				  'accent'
	  );
	  badge.setAttribute('title', `Total QSOs with ${park_num}`);
	  badge.innerHTML = `${hunted.qsos}`;
          objs.badges.append(badge);
        }else if( ! OPTIONS.enableNeededLocations ){
	  badge.classList.add('warning');
	  badge.setAttribute('title', `No QSOs with ${park_num}`);
	  badge.innerHTML = `0`; // atno
          objs.badges.append(badge);
        }
      }
    } // OPTIONS.enableNeededParks

//-------------------------------------
    if( OPTIONS.enableNeededOps && ! objs.call.innerHTML.match(/OPS_HUNTED/) ){
      if( OPS_HUNTED[call] ){
        var huntedCt = OPS_HUNTED[call].ct;
	var badge = document.createElement('span');
	badge.setAttribute('data-marker','OPS_HUNTED');
	badge.classList.add('v-badge__badge');
	badge.style.position = 'relative';
	badge.style.marginRight = '5px';
	badge.style.top = '10px';
	badge.style.left = '-5px';
	badge.classList.add(
		  huntedCt % 50 >= 45 ? 'error'  // Operator to Operator levels are every 50, so warn when close
		: huntedCt >= 50 ? 'grey'
		: huntedCt       ? 'accent'
		:                  'warning'
	);
	badge.setAttribute('title', `Total QSOs with ${call}`);
	badge.innerHTML = `${huntedCt}`;
        objs.call.append(badge);
      }else{
	var badge = document.createElement('span');
	badge.setAttribute('data-marker','OPS_HUNTED');
        objs.call.append(badge);
      }
      var spotBtn =	  LISTMODE ? objs.call.closest("tr").querySelector(".v-btn--fab")
			: CARDMODE ? objs.btns.querySelector(".v-btn")
			: null ;
      if( spotBtn ){
        spotBtn.setAttribute('data-callsign', call);
        spotBtn.setAttribute('data-parknum', park_num);
	if( false && spotBtn.getAttribute('data-click') != 1 ){
          spotBtn.setAttribute('data-click', 1);
          spotBtn.addEventListener('click', (e2) => {
	      var c = e2.target.closest('.v-btn').getAttribute('data-callsign');
	      var p = e2.target.closest('.v-btn').getAttribute('data-parknum');
	      lookupOpHunted(c);
	      lookupParkHunted(p);
	      updateActivationBio(c, p);
      	  } ); // async update OPS_HUNTED stats
	}
      }
      var historyBtn = objs.historyBtn; // .querySelector(".v-btn") || objs.historyBtn;
      historyBtn.setAttribute('data-callsign', call);
      historyBtn.setAttribute('data-parknum', park_num);
      if( false && historyBtn.getAttribute('data-click') != 1 ){
	historyBtn.setAttribute('data-click', 1);
        historyBtn.addEventListener('click', (e2) => {
	      var c = e2.target.closest('.v-btn').getAttribute('data-callsign');
	      var p = e2.target.closest('.v-btn').getAttribute('data-parknum');
	      lookupOpHunted(c);
	      lookupParkHunted(p);
	      updateActivationBio(c, p);
      	} ); // async update OPS_HUNTED stats
      }
    } // OPTIONS.enableNeededOps

//-------------------------------------
    if( OPTIONS.enableNeededN1CC && ! objs.park_num.innerHTML.match(/N1CC_HUNTED/) ){
      if(N1CC_HUNTED.length){
        var hunted = N1CC_HUNTED.find( x => x.reference === park_num );
        if( hunted != null ){
	  var badge = document.createElement('span');
	  badge.setAttribute('data-marker','N1CC_HUNTED');
	  badge.classList.add('v-badge__badge');
	  badge.classList.add('blue');
	  badge.style.position = 'relative';
	  badge.style.marginRight = '5px';
	  badge.innerHTML = `n1cc x${hunted.bands}`;
//console.log('N1CC_HUNTED', N1CC_HUNTED );
//console.log('N1CC', [ park_num, band, hunted ] );
	  if( ! hunted.hasOwnProperty('bandsWorked') ){
//console.log('FETCHING...', park_num, hunted);
// This should no longer happen.
	    badge.setAttribute('title', `${band} - ERROR WITH N1CC_HUNTED cache`);
            objs.badges.append(badge);
//	    updateBandsWorked(park_num, hunted, band, objs, badge);
	  }else if( ! hunted["bandsWorked"][band] ){
	    badge.setAttribute('title', `${band} - Needed band for N1CC (${hunted["bandsWorkedStr"]})`);
            objs.badges.append(badge);
//console.log('N1CC-UPDATED[B]', [ objs.park_num.innerText, park_num, band, hunted, badge ] );
	  }else{
	    badge.classList.remove('blue');
	    badge.classList.add('grey');
	    badge.setAttribute('title', `${band} Band worked for N1CC (${hunted["bandsWorkedStr"]})`);
            objs.badges.append(badge);
	  }
        }
      }
    } // OPTIONS.enableNeededN1CC

//-------------------------------------
    if( // OPTIONS.enableQsy &&
	    ! objs.freq.innerHTML.match(/QSY_BTN/) ){
//console.log("SPOT", [ call, freq, band, modeQsy, modeLog, park_num, park_region, park_name ] );
//console.log("SPOT", {"call":call, "freq":freq, "band":band, "modeQsy":modeQsy, "modeLog":modeLog, "park_num":park_num, "park_region":park_region, "park_name":park_name } );
      var x = document.createElement('span');
      x.setAttribute('data-marker', 'QSY_BTN' );
      var qsyLabel = LISTMODE ? 'QSY' : 'QSY '+freqLabel;
      var qsyStyle = btnStyle;
      qsyStyle += LISTMODE ? '' : 'min-width:80%;';
      var btnType="button";
      var btnClass = "ma-2 v-btn v-btn--is-elevated v-btn--has-bg theme--light v-size--default";
      if( !OPTIONS.enableQsy ){
	btnType="radio";
	qsyLabel = LISTMODE ? 'INFO' : 'INFO '+freqLabel;
	qsyStyle = "transform:scale(2.5); margin-right:10px;";
	btnClass = "";
      }
      x.innerHTML = `
	<input id="btn_qsy"
	    data-callsign="${call}"
	    type='${btnType}'
	    value='${qsyLabel}'
	    style="${qsyStyle}"
	    title="QSY ${freqLabel} ==> ${call} ${band} ${modeLog}"
	    data-freq="${freq}"
	    data-band="${band}"
	    data-mode="${modeQsy}"
	    data-freqorig="${freqOrig}"
	    data-spotKey="${spotKey}"
	    data-parknum="${park_num}"
	    class="${btnClass}"
	    ${ call == selectedcall ? 'CHECKED' : '' }
	    />
	    ${ CARDMODE && ! OPTIONS.enableQsy ? freqOrig : '' }
	`;
//console.log( LISTMODE, OPTIONS.enableQsy, freq, freqLabel, freqOrig, qsyLabel );
      x.addEventListener('click', (e2) => {
	if( !OPTIONS.enableQsy ){
	      return;
        }
	if( scanning && scan_call != e2.target.getAttribute('data-callsign') ){
	    // halt scanning if we're clicking on something out of order.
	    document.getElementById('btn_scan').click();
	}
	var freq = e2.target.getAttribute('data-freq');
	var mode = e2.target.getAttribute('data-mode');
	scan_call = e2.target.getAttribute('data-callsign'); // have scan pick up from here
	var url;
	var baseURL = OPTIONS.qsyConfig.proto + "://" + OPTIONS.qsyConfig.host + ":" + OPTIONS.qsyConfig.port;
	var baseParams = '';
//console.log(OPTIONS);
	if( OPTIONS.qsyConfig.target_host ){ baseParams += '&__host='+OPTIONS.qsyConfig.target_host; }
	if( OPTIONS.qsyConfig.target_port ){ baseParams += '&__port='+OPTIONS.qsyConfig.target_port; }
	if( OPTIONS.qsyConfig.method == 'commander' ){
          url = baseURL + "/dxlab/commander/CmdSetFreqMode"
			// param order matters here!
			+ "?xcvrfreq="+freq // kHz
			+ "&xcvrmode="+mode
			+ "&preservesplitanddual=N"
			+ baseParams;
	}else if( OPTIONS.qsyConfig.method == 'omnirig' ){
          url = baseURL + "/omnirig/qsy"
				+ "?freq=" + (freq*1000) // kHz (pota) => Hz (omnirig)
				+ "&mode=" + mode
				+ baseParams;
	}else if( OPTIONS.qsyConfig.method == 'aclog' ){
	  // ACLog does freq & mode in separate calls
          url = baseURL + "/aclog/changefreq"
				+ "?value=" + (freq/1000.0) // kHz (pota) => MHz (aclog)
				+ "&suppressmodedefault=TRUE"
				+ baseParams;
	  _fetch( url, { mode: 'no-cors' } );  // make 1st call ...
	  // ... and set url for second call, to be made below.
          url = baseURL + "/aclog/changemode"
				+ "?value=" + mode
				+ baseParams;
	}else if( OPTIONS.qsyConfig.method == 'logger32' ){
          url = baseURL + "/logger32/qsy"
				+ "?APP_SET_FREQ_MODE=" + freq // kHz
							+ "|" + mode
				+ baseParams;
	}
//console.log("URL:", url);
	_fetch( url, {});//{ mode: 'no-cors' } );
      });
      if( LISTMODE ){
         objs.freq.prepend(x);
      }else{
        objs.freq.innerHTML = '';
        objs.freq.append(x);
      }
    } // OPTIONS.enableQsy
    objs.freq.style.textDecoration = '';
    if( OPTIONS.enableBandEdgeWarnings && ! objs.freq.innerHTML.match(/BAND_EDGE/) ){
	var y = document.createElement('span');
	y.setAttribute('data-marker', 'BAND_EDGE' );
	if( isUSGeneralOutOfBand ){
		y.innerHTML += " {!EXTRA!}";
	}
	if( isSSBUSBandEdges ){
		y.innerHTML += " {!EDGE!}";
	}
	if( isUSGeneralOutOfBand || isSSBUSBandEdges ){
		objs.freq.append(y);
		objs.freq.style.textDecoration = 'line-through';
	}
    } // OPTIONS.enableBandEdgeWarnings

//-------------------------------------
    if( OPTIONS.enableLogging && ! objs.btns.innerHTML.match(/LOG_BTN/) ){
      var y = document.createElement('span');
      y.setAttribute('data-marker', 'LOG_BTN' );
      y.innerHTML = `
	<input
	    type=button
	    value="QSL"
	    style="${btnStyle}; min-width:auto; padding-left: 5px; padding-right: 5px"
	    title="LOG ${call} @${park_num} ${band} ${modeLog}"
	    class="ma-2 v-btn v-btn--is-elevated v-btn--has-bg theme--light v-size--default orange accent-1 black--text"
	    data-freq="${freq}"
	    data-band="${band}"
	    data-mode="${modeLog}"
	    data-call="${call}"
	    data-park_num="${park_num}"
	    data-park_region="${park_region}"
	    data-park_name="${park_name}"
	    data-marker="QSL_BTN"
	    />
	`;
      y.addEventListener('mouseover', (e2) => {
	var p = e2.target.getAttribute("data-park_num");
	if( ! PARK_INFO[p] ){
	  lookupParkHunted(p);
	}
      });
      y.addEventListener('click', (e2) => {
	var p = e2.target.getAttribute("data-park_num");
	if( ! PARK_INFO[p] ){
	  lookupParkHunted(p);
	}
      });
	    // TODO --add double-click protection
      y.addEventListener('click', (e2) => {
	// TODO .. need to find callsign cell and cross-check the call to trap the error where QSL button is shifted wrt to spot row.
	var freq = e2.target.getAttribute('data-freq');
	var band = e2.target.getAttribute('data-band');
	var mode = e2.target.getAttribute('data-mode');
	var call = e2.target.getAttribute('data-call');
	var park_num = e2.target.getAttribute('data-park_num');
	var park_region = e2.target.getAttribute('data-park_region');
	var park_name = e2.target.getAttribute('data-park_name');
	var parkInfo = ( PARK_INFO[park_num] ? PARK_INFO[park_num]["info"] : {} ) || {};  // Will be missing park info if the QSY/radio selection was not used.
	var baseURL = OPTIONS.loggingConfig.proto + "://" + OPTIONS.loggingConfig.host + ":" + OPTIONS.loggingConfig.port;
	var baseParams = '';
	if( OPTIONS.loggingConfig.target_host ){ baseParams += '&__host='+OPTIONS.loggingConfig.target_host; }
	if( OPTIONS.loggingConfig.target_port ){ baseParams += '&__port='+OPTIONS.loggingConfig.target_host; }
	var additionalFields = '';
	Object.keys( OPTIONS.loggingConfig.additionalADIFFields ).forEach(function (key){
	  var value = OPTIONS.loggingConfig.additionalADIFFields[key];
	  value = value.replaceAll('%MYCALL%', MYCALL );
	  value = value.replaceAll('%PARKREF%', '__PARK_NUMS__' ); // park_num
	  value = value.replaceAll('%GRID4%', parkInfo.grid4 || '' );
	  value = value.replaceAll('%GRID6%', parkInfo.grid6 || '');
	  additionalFields += "&" + key + "=" + value;
	});
	if( OPTIONS.loggingConfig.parkNumADIFField
		&& OPTIONS.loggingConfig.parkNumADIFField.trim()
		&& ! OPTIONS.loggingConfig.parkNumADIFField.match(/__DEPRECATED__/)
	    ){
	  additionalFields += "&"
		+ OPTIONS.loggingConfig.parkNumADIFField.trim().split(/[, ]+/).map(k=> k+"=__PARK_NUMS__").join("&")
	  ;
	}
        var url;
        var ping_url = '';
	var commentStr = "[POTA"
		+ " " + park_num
		+ " " + park_region
		+ " " + (parkInfo.grid6||'')
		+ " " + park_name
		+ "] __COMMENTS__";
	if( OPTIONS.loggingConfig.method == 'dxkeeper' ){
		//  http://www.dxlabsuite.com/dxkeeper/DXKeeper%20TCPIP%20Messages%20v1.pdf
          url = baseURL + "/dxlab/dxkeeper/log"
	    + "?" + "CALL=__CALL__"
	    + "&RST_SENT=__RST_SENT__"
	    + "&RST_RCVD=__RST_RCVD__"
	    + "&FREQ=" + (freq / 1000.0)  // kHz (pota) => MHz (adif)
	    + "&FREQ_RX=" + (freq / 1000.0)  // kHz (pota) => MHz (adif)
	    + "&MODE=" + mode
	    + "&COMMENT=" + commentStr
	    + "&QSO_DATE=__DATE_YYYYMMDD__" // YYYYMMDD 
	    + "&TIME_ON=__TIME_HHMMSS__" //   HHMMSS
	    + "&TX_PWR=__TX_PWR__"
	    + "&RX_PWR=__RX_PWR__"
	    + additionalFields
	    + baseParams
          ;
	  ping_url = baseURL + "/dxlab/dxkeeper/ping" + "?" + baseParams;
	}else if( OPTIONS.loggingConfig.method == 'logger32' ){
		//  https://www.logger32.net/files/Logger32_v4_User_Manual.pdf	Sections 31 & 32
          url = baseURL + "/logger32/log"
	    + "?" + "CALL=__CALL__"
	    + "&RST_SENT=__RST_SENT__"
	    + "&RST_RCVD=__RST_RCVD__"
	    + "&FREQ=" + (freq / 1000.0)  // kHz (pota) => MHz (adif)
	    + "&FREQ_RX=" + (freq / 1000.0)  // kHz (pota) => MHz (adif)
	    + "&MODE=" + mode
	    + "&COMMENT=" + commentStr
	    + "&QSO_DATE=__DATE_YYYYMMDD__" // YYYYMMDD 
	    + "&TIME_ON=__TIME_HHMMSS__" //   HHMMSS
	    + "&RX_PWR=__RX_PWR__"
	    + "&BAND=" + band // Logger32 doesn't seem to calculate this from the freq fields
	    + additionalFields
	    + baseParams
          ;
	  ping_url = baseURL + "/logger32/ping" + "?" + baseParams;
	}else if( OPTIONS.loggingConfig.method == 'log4om' ){
          url = baseURL + "/log4om/log"
	    + "?" + "CALL=__CALL__"
	    + "&RST_SENT=__RST_SENT__"
	    + "&RST_RCVD=__RST_RCVD__"
	    + "&FREQ=" + (freq / 1000.0)  // kHz (pota) => MHz (adif)
	    + "&FREQ_RX=" + (freq / 1000.0)  // kHz (pota) => MHz (adif)
	    + "&MODE=" + mode
	    + "&COMMENT=" + commentStr
	    + "&QSO_DATE=__DATE_YYYYMMDD__" // YYYYMMDD 
	    + "&TIME_ON=__TIME_HHMMSS__" //   HHMMSS
	    + "&TX_PWR=__TX_PWR__"
	    + "&RX_PWR=__RX_PWR__"
//	    + "&APP_L4ONG_QSO_AWARD_REFERENCES=" + '[{"AC":"POTA","R":"' + park_num + '","G":"' + park_region + '","SUB":[],"GRA":[]}]'
//	    + "&APP_L4ONG_QSO_AWARD_REFERENCES=" + JSON.stringify([ { "AC":"POTA", "R":park_num, "G":park_region, "SUB":[], "GRA":[] } ])
	    + "&APP_L4ONG_QSO_AWARD_REFERENCES=__LOG4OM_PARKS__"
	    + additionalFields
	    + baseParams
          ;
	  ping_url = baseURL + "/log4om/ping" + "?" + baseParams;
	}else if( OPTIONS.loggingConfig.method == 'aclog' ){
		//  http://www.n3fjp.com/help/api.html
          url = baseURL + "/aclog/updateandlog"
	    + "?" + "CALL=__CALL__"
	    + "&MODE=" + mode
	    + "&FREQ=" + (freq / 1000.0)  // kHz (pota) => MHz (aclog)
	    + "&RSTR=__RST_RCVD__"
	    + "&RSTS=__RST_SENT__"
	    + "&COMMENTS=" + commentStr
	    + "&DATE=" + new Date().toISOString().slice(0 ,10).replaceAll("-","/") // YYYY/MM/DD 
	    + "&TIMEON=" + new Date().toISOString().slice(11,16) //   HH:MM
	    + "&POWER=__TX_PWR__"
		// __RX_PWR__   not suppported in ACLog?
	    + additionalFields
	    + baseParams
          ;
	}else if( OPTIONS.loggingConfig.method == 'qrz' ){  // API, not ham-apps-proxy
          url = // no ham-apps-proxy baseURL ... target will be in actual call; this is purely query params.
	      "CALL=__CALL__"
	    + "&RST_SENT=__RST_SENT__"
	    + "&RST_RCVD=__RST_RCVD__"
	    + "&FREQ=" + (freq / 1000.0)  // kHz (pota) => MHz (adif)
	    + "&FREQ_RX=" + (freq / 1000.0)  // kHz (pota) => MHz (adif)
	    + "&MODE=" + mode
	    + "&COMMENT=" + commentStr
	    + "&QSO_DATE=__DATE_YYYYMMDD__" // YYYYMMDD 
	    + "&TIME_ON=__TIME_HHMMSS__" //   HHMMSS
	    + "&TX_PWR=__TX_PWR__"
	    + "&RX_PWR=__RX_PWR__"
	    + additionalFields
	    // note: baseParams only used for ham-apps-proxy
          ;
	}
	var spotBtn =	  LISTMODE ? e2.target.closest("tr").querySelector(".v-btn--fab")
			: CARDMODE ? e2.target.closest(".v-card__actions").querySelector(".v-btn")
			: null ;
//	var historyBtn =  LISTMODE ? e2.target.closest("tr").querySelector(".v-btn--round") // TODO test user-avatars
//			: CARDMODE ? e2.target.closest(".v-card__actions").querySelector(".v-btn") // TODO
//			: null ;
	if( 0 ){   // Old direct-LOG btn, without hijacking the Re-Spot modal.
		url = url.replaceAll('__RST_RCVD__', mode == "SSB" ? "59" : "599" );
		url = url.replaceAll('__RST_SENT__', mode == "SSB" ? "59" : "599" );
		// TODO -- add TX_PWR tags above for each logger
		url = url.replaceAll('__TX_PWR__', '' );
		url = url.replaceAll('__RX_PWR__', '' );
		url = url.replaceAll('__COMMENTS__', '' );
		url = url.replaceAll('__CALL__', call );
		url = url.replaceAll('__DATE_YYYYMMDD__', new Date().toISOString().slice(0 ,10).replaceAll("-","") );
		url = url.replaceAll('__TIME_HHMMSS__',   new Date().toISOString().slice(11,19).replaceAll(":","") );
		url = url.replaceAll('__LOG4OM_PARKS__', JSON.stringify([ { "AC":"POTA", "R":park_num, "G":park_region, "SUB":[], "GRA":[] } ]) );
		url = url.replaceAll('__PARK_NUMS__', park_num );
		_fetch( url, { mode: 'no-cors' } );
		if( OPTIONS.loggingConfig.autoReSpot ){
			spotBtn.click();
		}
	}else{
//		historyBtn.click();
//		var historyModal = document.querySelector(".v-dialog .v-dialog--active");
//		historyModal.style.left = '0px';
//		historyModal.style.position = 'absolute';
		spotBtn.click();
		var clone = e2.target.cloneNode(true); // make a copy to inject into the spot dialog
		setTimeout(() => { hijackSpotForm(url,call,park_num,park_region,mode, clone, ping_url ); }, 500 );  // slight delay, to give time for the spot dialog to open.  Must be long enough, or the errant-spots bug occurs!
	}
      });
      objs.btns.append( y );
    } // OPTIONS.enableLogging

    if( ! objs.btns.innerHTML.match(/TOGGLE_BTN/) ){
      var y = document.createElement('span');
      y.setAttribute('data-marker', 'TOGGLE_BTN' );
      var toggle = TOGGLES[spotKey] || 0;
      y.innerHTML = `
	<input
	    type=button
	    value="~"
	    style="${btnStyle};background-color:lightgrey !important; min-width: unset;"
	    title="Custom row toggle"
	    class="ma-2 v-btn v-btn--is-elevated v-btn--has-bg theme--light v-size--default accent-1 black--text"
	    data-call="${call}"
	    data-spotKey="${spotKey}"
	    data-toggle="${toggle}"
	    />
	`;
      var bgcolor = toggleColor(toggle);
      setBackgroundForRow(objs.btns, bgcolor);
      y.addEventListener('click', (e2) => {
		var call = e2.target.getAttribute('data-call');
		var spotKey = e2.target.getAttribute('data-spotKey');
		var toggle = e2.target.getAttribute('data-toggle');
		toggle = ( toggle + 1 ) % 3;
		TOGGLES[spotKey] = toggle;
		var bgcolor = toggleColor(toggle);
		e2.target.setAttribute('data-toggle', toggle );
		setBackgroundForRow(e2.target, bgcolor);
		var toggleBtnColor = bgcolor || 'lightgrey';
		e2.target.style.setProperty( 'background-color', toggleBtnColor, (toggleBtnColor?'important':'') );
		sendChromeMessage("set__TOGGLES",  TOGGLES );
      });
      objs.btns.append( y );
    } // TOGGLE_BTN

    if( OPTIONS.enableIconQRZ && ! objs.extLinks.innerHTML.match(/QRZ_LINK/) ){
	var a = document.createElement('a');
	a.setAttribute('href','https://qrz.com/db/'+call);
	a.setAttribute('title','QRZ.com');
	a.setAttribute('target','_blank');
        a.setAttribute('data-marker', 'QRZ_LINK');
	a.innerHTML = '<img style="width:20px; vertical-align:middle;margin-left:3px" src="' + chrome.runtime.getURL("images/qrz.ico") + '" />';
	objs.extLinks.prepend(a);
    }
    if( OPTIONS.enableIconRBN && ! objs.extLinks.innerHTML.match(/RBN_LINK/)  && modeLog == 'CW' ){
	var a = document.createElement('a');
	a.setAttribute('href','http://www.reversebeacon.net/dxsd1/dxsd1.php?f=0&t=dx&c='+call);
	a.setAttribute('title','RBN');
	a.setAttribute('target','_blank');
        a.setAttribute('data-marker', 'RBN_LINK');
	a.innerHTML = '<img style="width:20px; vertical-align:middle" src="' + chrome.runtime.getURL("images/rbn.ico") + '" />';
	objs.extLinks.prepend(a);
    }
    if( OPTIONS.enableIconPSKReporter && ! objs.extLinks.innerHTML.match(/PSKR_LINK/)  && modeLog != 'CW' && modeLog != 'SSB' ){
	var a = document.createElement('a');
	a.setAttribute('href','https://pskreporter.info/pskmap?callsign='+call);
	a.setAttribute('title','PSKR');
	a.setAttribute('target','_blank');
        a.setAttribute('data-marker', 'PSKR_LINK');
	a.innerHTML = '<img style="width:20px; vertical-align:middle" src="' + chrome.runtime.getURL("images/pskreporter.ico") + '" />';
	objs.extLinks.prepend(a);
    }
    if( ! objs.extLinks.innerHTML.match(/SPACER_LINK|PSKR_LINK|RBN_LINK/) ){
	var a = document.createElement('span');
        a.setAttribute('data-marker', 'SPACER_LINK');
	a.style.width = "23px";
	a.style.display = "inline-block";
	objs.extLinks.prepend(a);
    }
    if( ! objs.extLinks.innerHTML.match(/SPACER2_LINK|QRZ_LINK/) ){
	var a = document.createElement('span');
        a.setAttribute('data-marker', 'SPACER2_LINK');
	a.style.width = "20px";
	a.style.display = "inline-block";
	objs.extLinks.prepend(a);
    }

      if( call == selectedcall ){
	setBackgroundForRow(objs.btns, COLOR1);
	updateBio(OPS_HUNTED[call]);
      }else if( TOGGLES[spotKey] ){
	var bgcolor = toggleColor( TOGGLES[spotKey] );
	setBackgroundForRow(objs.btns, bgcolor);
      }else{
	setBackgroundForRow(objs.btns, "");
      }
      var toggleBtnColor = TOGGLES[spotKey] ? toggleColor( TOGGLES[spotKey] ) : 'lightgrey';
      row.querySelector('[data-marker="TOGGLE_BTN"]').querySelector('input').style.setProperty( 'background-color', toggleBtnColor, (toggleBtnColor?'important':'') );

///////////////////////////////////////
    } // rows

    var filterHtml = `
	<form style="background-color: lightgrey; padding:8px; width: fit-content; border-radius:10px">
		<b>HIDE SPOTS:</b>
    `;
    //
    for( let attributeKey of ['modes','bands','programs','regions'] ){
      var whitespace = attributeKey == 'regions' ? 'inherit' : 'nowrap';
      if( attributeKey == 'regions' ){
	      filterHtml += '<br/>';
      }
      if( attributeKey == 'regions' ){
	      filterHtml += `<input id="spot_filter_details_toggle" type="button" value="~" data="${SPOT_FILTER_DETAILS_TOGGLE}" style="border:1px solid black; border-radius:5px; margin-right:5px; padding:3px; background-color:grey">`;
      }
      filterHtml += `<span style="border:1px solid grey; border-radius:5px; margin-right:5px; padding-left:3px; padding-right:5px; white-space: ${whitespace}">`;
      var kk = Object.keys(SPOT_ATTRIBUTES[attributeKey]).sort();
      if( attributeKey == 'bands' ){
	kk = kk.sort( (a,b) => {
		var x = a.replace(/(M|cm)$/i, '');
		if( a.match(/cm/i) ){ x /= 1000.0; }
		var y = b.replace(/(M|cm)$/i, '');
		if( b.match(/cm/i) ){ y /= 1000.0; }
		return x - y;
	} );
      }
      for ( let k of kk ){
	var style = "";
        if( attributeKey == 'regions' ){
        	if( k.match(/\+\s*\d+/) ){
			continue;
		}
		var loc = k.replace(/-.+/, '');
        	if( SPOTS_TO_HIDE[loc] ){
			continue;
		}
        	var hunted = LOCATIONS_HUNTED.find( x => x.locationDesc === k );
        	if( ! hunted ){
			style = "background-color: orange; ";
		}
	}
        if( attributeKey == 'programs' ){
        	var hunted = LOCATIONS_HUNTED.find( x => x.locationDesc.replace(/-.+/,'') === k );
        	if( ! hunted ){
			style = "background-color: orange; ";
		}
	}
	var slotsHtml = Object.keys(SPOT_ATTRIBUTES[attributeKey][k]).sort().join('\n');
	var checked = SPOTS_TO_HIDE[k] ? 'CHECKED' : '';
	filterHtml += `&nbsp;&nbsp;<span style="${style}; display:inline-table" title="${slotsHtml}" ><input id="spots_to_hide" type="checkbox" value="${k}" ${checked} /> ${k.replace(/^US-/,'')}`;
	if( attributeKey == 'regions' ){
	  filterHtml += `<br/><span id="spot_filter_details" style="display:${SPOT_FILTER_DETAILS_TOGGLE?'':'none'}">${slotsHtml.replace(/\s+/g,'<br/>')}</span>`;
	}
	filterHtml += `</span>`;
      }
      filterHtml += `</span>`;
    }
    //
    filterHtml += `
	</form>
    `;
    var filterObj = document.getElementById('filter_container');
    filterObj.innerHTML = filterHtml;
    filterObj.style.display = OPTIONS.enableSpotFilters ? '' : 'none';
    Array.from( filterObj.querySelectorAll('input#spots_to_hide') , e => {
      e.addEventListener('click', (e2) => {
	      var k = e2.target.value;
	      var v= e2.target.checked;
	      updateFilterHide(k,v);
      });
    });
    document.querySelector('#spot_filter_details_toggle').addEventListener('click', (e) => {
      var toggle = e.target.getAttribute('data') > 0 ? 0 : 1;
      SPOT_FILTER_DETAILS_TOGGLE = toggle;
      e.target.setAttribute('data', toggle);
      Array.from( document.querySelectorAll('#spot_filter_details') , (e2) => {
	      e2.style.display = toggle ? '' : 'none';
      });
    });


    Array.from( document.querySelectorAll('input#btn_qsy') , e => {
      e.addEventListener('click', (e2) => {
	var callsign = e2.target.getAttribute("data-callsign");
	var spotKey = e2.target.getAttribute("data-spotKey");
	var p = e2.target.getAttribute("data-parknum");
	lookupOpHunted(callsign); // async update OPS_HUNTED stats
	lookupParkHunted(p);
	updateActivationBio(callsign, p);
	//console.log('CLICKED QSY', callsign );
//        setCurrentCallsignBackground('inherit');
        chrome.runtime.sendMessage({ 
            message: "set__selectedcall",
            payload: callsign
          }, response => {
            if (response.message === 'success') {
            }
        });
//	setBackgroundForCallsign(selectedcall3, 'inherit');
	if( selectedcall != callsign ){
	setBackgroundForCallsign(selectedcall, '');
		selectedcall3 = selectedcall2;
		selectedcall2 = selectedcall;
		selectedcall  = callsign;
        setCurrentCallsignBackground(COLOR1);
	}
//        setBackgroundForCallsign(selectedcall2, COLOR2);
//        setBackgroundForCallsign(selectedcall3, COLOR3);
      });
      setCurrentCallsignBackground(COLOR1);
//      setBackgroundForCallsign(selectedcall2, COLOR2);
//      setBackgroundForCallsign(selectedcall3, COLOR3);
    });

    observe(); // turn back on 
}); // new MutationObserver
}

// TODO: callsign lookup (DXV, Pathfinder, DXK) ;  needs DDE support in HTTP/DXLab bridge
// // TODO: Prompt for exchanges
// localspots to Spotcollector

var observe = ()=> {
  observer.observe(MAINTABLE, {
    childList: true,
    characterData: true,
    subtree: true,
  });
};
observe();
MAINTABLE.append(" "); // touch, to trigger initial update


    Array.from( document.querySelectorAll('div.v-input--selection-controls__input') , e => {
      e.addEventListener('click', (e2) => {
	setTimeout(() => {
	  detectLayout();
	  MAINTABLE.append(" "); // touch, to trigger initial update
	}, 250 );
      });
    });




function checkKey(e) {
    e = e || window.event;
//    var scan_btn = document.getElementById('btn_scan');
    if( OPTIONS.enableQsy  // rig control enabled
	    && ! document.querySelector('.v-overlay') // not in a modal, e.g. re-spot, history, log
      ){
	// 37=Left, 38=Up, 39=Right, 40=Down
	if( false ){
	} else if (e.keyCode == '37' && ! e.ctrlKey && ! e.shiftKey && ! e.altKey ) { // left arrow
//	    scanning = 1;
//	    scanNext(-1);
//	    scanning = 0;
	    startScan(-1);
	    stopScan();
	    return false;
    	} else if (e.keyCode == '39'  && ! e.ctrlKey && ! e.shiftKey && ! e.altKey ) { // right arrow
//	    scanning = 1;
//	    scanNext();
//	    scanning = 0;
	    startScan();
	    stopScan();
	    return false;
    	} else if ( e.keyCode == '32' &&   e.ctrlKey && ! e.shiftKey && ! e.altKey ) { // ctrl-space
	    document.getElementById('btn_scan').click();
	    return false;
	}
    }
}
document.onkeydown = checkKey;


if( !MYCALL ){
	document.querySelector('#label_update').innerHTML = 'PLEASE SIGN IN TO POTA.APP ACCOUNT';
	document.querySelector('#label_update').style.display = '';
}

